﻿/**
* The gdsQuery instance of gdsLibrary
*/
(function() {

    var proximitySensorUrn = 'urn:gds:chn:Peripherals:Proximity';
    var temperatureCalibrationUrn = 'urn:gds:chn:Peripherals:Temperature';
    var timers = {};
    var tsxReadyHandler = $.Callbacks();
    var isReady = false;

    /**
    * Adds an empty switchTimes array if the timer does not comrprise switchTimes
    * 
    * @param {object} timerObject Timer object from the gds api
    */
    function emptySwitchTimesFix(timerObject) {
        if (timerObject.timer && !timerObject.timer.switchTimes) {
            timerObject.timer.switchTimes = [];
        }
    }

    /**
    * Updates the local timer object if an according event is received and notifies the timer module.
    * 
    * @param {string} id Id of the gds timer object
    */
    function onConfigChange(event) {
        var id = event.configuration.id;
        if (id in timers) {
            gds$.getConfigById(id, function (response) {
                if (timers[id].timer) {
                    timers[id].timer.switchTimes.splice(0, timers[id].timer.switchTimes.length);
                }
                emptySwitchTimesFix(response.object);
                $.extend(true, timers[id], response.object);
                tsx.timer.update(id, response.object);
            });
        }
    }


    var tsxGdsUtility = function () {
        this.init();
    };

    tsxGdsUtility.prototype = {
        init: function () {
            gds$.loglevel = 0;
            gds$.ready(function() {
                isReady = true;
                tsxReadyHandler.fire();
            });
            gds$.addConfigurationHandler(onConfigChange);

        },
		ready: function (handler) {
		    tsxReadyHandler.add(handler);
		},
		getMetadataValue: function (callback, metadata, key, defaultValue) {
            metadata.forEach(function (element) {
                if (element["key"] === key) {
                    callback(element["value"]);
                } else {
                    if (defaultValue != undefined) {
                        callback(defaultValue);
                    }
                }
            });
        },
        setProximitySensorSensitivity: function (sensitivity) {
            var object = {
                id: gds$.getId(proximitySensorUrn),
                metadata: [
                {
                    key: "Sensitivity",
                    value: sensitivity
                }]
            };
            gds$.setConfigObject(object);

        },
        getProximitySensorSensitivity: function (callback) {
            var id = gds$.getId(proximitySensorUrn);
            gds$.getConfigById(id, function (response) {
                tsxGdsUtility$.getMetadataValue(callback, response.object.metadata, "Sensitivity", "off");
            });
        },
        setTemperatureCalibration : function (sensitivity) {
            var object = {
                id: gds$.getId(temperatureCalibrationUrn),
                metadata: [
                {
                    key: "calibration",
                    value: sensitivity
                }]
            };
            gds$.setConfigObject(object);
        },
        getTemperatureSetPoint: function (rtcId, callback) {
			tsxGdsUtility$.ready(function(){
			    gds$.getConfigById(rtcId, function (response) {
			        if (response.object.metadata) {
			            var setPoint = new Array();
			            try {
			                for (var i = 0; i < response.object.metadata.length; i++) {
			                    setPoint[response.object.metadata[i].key] = parseFloat(response.object.metadata[i].value).toFixed(1);
			                }
			            } catch (e) {
			                console.log(e);
			            }
			            callback(setPoint);
			        }
				});
			});
        },
        setTemperatureSetPoint: function (rtcId, id, temperatureValue) {
            var object = {
                id: rtcId,
                metadata: [
                {
                    key: id,
                    value: temperatureValue
                }]
            };
            gds$.setConfigObject(object);
        },
        /**
         * Returns the current list of timers for the specified timer object
         * 
         * @param {string} id Id of the gds timer object
         * @return Array of timer data
         */
        getTimers: function (id) {
            try {
                return timers[id].timer.switchTimes;
            } catch (e) {
                return [];
            }
        },
        /**
         * Sends a message to set a specific timer
         * 
         * @param {string} id Id of the gds timer object
         * @param {object} timerInfo comprises the data of the timer
         */
        setTimer: function (id, timerInfo) {

            var switchTime = {
                id: id,
                timer: {
                    switchTimes: [timerInfo]
                }
            };

            gds$.setConfigObject(switchTime);
        },
        /**
         * Sends a delete message for one or more timers
         * 
         * @param {string} id Id of the gds timer object
         * @param {array} timerIds array with the ids of all timers whitch shall be deleted
         */
        deleteTimers: function (id, timerIds) {
            var queryObject = {
                id: id,
                timer: {
                    switchTimes:[]
                }

            }
            timerIds.forEach(function(timerId) {
                queryObject.timer.switchTimes.push({
                    index: timerId.toString(),
                    remove: "true"
                });
            });
            gds$.setConfigObject(queryObject);
        },
        /**
        * Sends a delete message for a specific number of timers and delete all of its switchTimes object (10)
        *
        * @param {string} startId id of the first timer to start with
        * @param {string} count   number of timers to be deleted
        */
        deleteAllTimers: function (startId, count) {
            gds$.emitConfigurationChangeBegin();
            var sId = parseInt(startId);
            gds$.disableConfigRange(sId, sId + (count - 1) * globals.timer.rfcClockOffset);
            gds$.emitConfigurationChangeEnd();

            var switchTimes = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9];

            for (var i = sId; i < sId + count * globals.timer.rfcClockOffset; i += globals.timer.rfcClockOffset) {
                this.deleteTimers(i, switchTimes);             
            }
        },

        /**
        * This function sends a delete message for all RTC timers (max. 28)
        */
        deleteRtcTimers: function () {
            var switchTimes = [];
            for (var i = 0; i < 28; i++) {
                switchTimes[i] = i+1;
            }
    
            this.deleteTimers(40200, switchTimes);
            this.setTimerState(40200, false);
        },
        /**
         * Returns the current enabled state of a timer object
         * 
         * @param {string} id Id of the gds timer object
         * @return enabled state of the timer, either 'true' or 'false'
         */
        getTimerState: function (id) {
            try {
                return timers[id].enabled === 'true';
            } catch (e) {
                return false;
            }
        },
        /**
         * Sets the current enabled state of a timer object
         * 
         * @param {string} id Id of the gds timer object
         * @param {boolean} enabled new enabled state of the timer object
         */
        setTimerState: function (id, enabled) {
            gds$.setConfigObject({ id: id, enabled: enabled.toString() });
        },

        /**
         * Initializes a timer object with the according scenes and registers the timer id
         * for updates.
         * 
         * @param {string} id Id of the gds timer object
         * @param {array} scenes array with scenes for the timer object
         */
        registerTimer: function (id, timerFunctions) {
            var datapoints = [];
            var sceneSet = {
                datapoints: [],
                scenes: []
            };

            timerFunctions.forEach(function (timerFunction) {
                timerFunction.values.forEach(function (value) {
                    sceneSet.scenes.push({
                        number: (sceneSet.scenes.length + 1).toString(),
                        name: timerFunction.name + '-' + value,
                        datapoints: [{ id: timerFunction.datapoint, value: value }]
                    });
                    if (datapoints.indexOf(timerFunction.datapoint) < 0) {
                        datapoints.push(timerFunction.datapoint);
                    }
                });
            });
            datapoints.forEach(function (element) {
                sceneSet.datapoints.push({
                    id: element
                });
            });

            var initTimer = function () {
                gds$.setConfigObject({
                    id: id, timer: { switchTimes: [] }, sceneSet: sceneSet
                });
                gds$.getConfigById(id, function (response) {
                    if (response.object) {
                        emptySwitchTimesFix(response.object);
                        timers[id] = response.object;
                        tsx.timer.update(id, timers[id]);
                    }
                });
                tsxReadyHandler.remove(initTimer);
            };

            isReady ? initTimer() : this.ready(initTimer);
        },
        /**
        * Sends value to trigger splashscreen removal. Retries if the response conatains an error.
        */
        removeSplashScreen: function () {
            gds$.setValueWithFeedback("400403", "0", function(response) {
                if (response.error.code !== "0") {
                    setTimeout(tsxGdsUtility$.removeSplashScreen, 1000);
                }
            });
        }

    }; // End of gdsLibrary

    window.tsxGdsUtility$ = new tsxGdsUtility();
})();