﻿/**
 * The timer controller handles UI states and the functionality of the timer view.
 */
window.tsx.timer = new (function () {
    var module = {},
        I18nTexts = window.Gira.appLang.appI18nTexts;

    var timerModes = { list: 0, edit: 1, editTimer: 2, closed: 3, addTimer: 4 };
    var timerMode = timerModes.list;
    var previousMode;
    var isTimerEnabled;
    var timerActionNames;
    var currentTimerId;
    var timerIcons = {};
    var timerTypes = '';
    var dayTimeMin = moment('05:59', 'hh:mm');
    var dayTimeMax = moment('18:00', 'hh:mm');
    var dayStrings = [I18nTexts.timerDaysMon, I18nTexts.timerDaysTue, I18nTexts.timerDaysWed, I18nTexts.timerDaysThu, I18nTexts.timerDaysFri, I18nTexts.timerDaysSat, I18nTexts.timerDaysSun];
    var scrollPosition = 0;

    // handles generation and updates of the timer list
    var timerListModel = new function () {

        // Comprises the template data for each timer row
        this.timers = ko.observableArray();

        /**
         * Registers events for a timer list row
         *
         * @param {array} element Timer row element
         * @param {object} data Timer row template data
         */
        this.registerEvents = function (element, data) {
            var $element = $(element[1]);
            registerEventsForHorizontalSwipeableElement($element,
                function (event) {
                    if (timerMode === timerModes.list && isTimerEnabled) {
                        onTouchStart($element, event);
                    }
                },
                function () {
                    if (timerMode === timerModes.list && isTimerEnabled) {
                        onTouchEnd($element);
                    }
                },
                function () {
                    if (timerMode === timerModes.list && isTimerEnabled) {
                        editTimer(data);
                    }
                }
            );

            var $checkbox = $('.checkbox', $element);
            registerEventsForHorizontalSwipeableElement($('.checkbox-wrapper', $element), undefined, undefined, function () {
                if ($checkbox.attr('checked') == null || $checkbox.attr('checked') == undefined) {
                    $checkbox.attr('checked', true);
                } else {
                    $checkbox.attr('checked', null);
                }
                updateDeleteButton();
            });

            if (timerMode === timerModes.edit) {
                $('.checkbox, .timer-box', $element).addClass('edit-mode');
                $('.timer-arrow', $element).css('opacity', 0);
            }

            $('#timer-list-view').data('settingsSwiper').resizeFix(true);
        }

        /**
         * Formats the weekdays entry according to recurrence
         *
         * @param {string} recurrence recurrence of the choosen timer
         * @param {string} weekdays weekdays entry of the choosen timer
         * @returns {string} formatted recurrence string
         */
        this.formatDays = function (recurrence, weekdays) {
            if (recurrence === 'daily' && weekdays != undefined) {
                var dayArray = new Array(7);
                var daysString = '';
                var dayString = '';

                if (weekdays === '-------') {
                    return '---';
                }

                for (var i = 0; i < dayArray.length; ++i) {
                    if (weekdays.substring(i, i + 1) !== '-') {
                        dayArray[i] = '1';
                    } else {
                        dayArray[i] = '-';
                    };
                    dayString = '';
                    if (dayArray[i] === '1') {
                        dayString = dayStrings[i];
                        var isAnyPreviousDaySet = false;
                        var j = 0;

                        for (j = i; j > 0; j--) {
                            if (dayArray[i - j] === '1') {
                                isAnyPreviousDaySet = true;
                                break;
                            }
                        }

                        if (isAnyPreviousDaySet) {
                            var isBothPreviousTwoDaysSet = true;
                            for (j = 1; i - j >= 0 && j <= 2; j++) {
                                if (dayArray[i - j] === '-') {
                                    isBothPreviousTwoDaysSet = false;
                                    break;
                                }
                            }
                            if (i > 1 && isBothPreviousTwoDaysSet) {
                                daysString = daysString.substring(0, daysString.length - 3);
                                dayString = '-' + dayString;
                            } else if (dayArray[i - 1] === '1') {
                                dayString = '-' + dayString;
                            } else if (dayArray[i - 1] === '-') {
                                dayString = ', ' + dayString;
                            }
                        }
                    }
                    daysString = daysString.concat(dayString);
                }
                return daysString;
            } else if (recurrence === 'daily') {
                return 'Mo-So';
            } else if (recurrence === 'once') {
                return 'einmalig';
            } else {
                return '---';
            }
        }

        this.formatTime = function (time) {
            return moment(time, 'hh:mm:ss').format('HH:mm');
        }

        this.formatAction = function (action) {
            return timerActionNames[action];
        }

        this.updateModel = function() {
            var swiper = $('#timer-list-view').data('settingsSwiper');
            if (swiper) {
                scrollPosition = swiper.getWrapperTranslate('y');
                this.timers.valueHasMutated();
                swiper.setWrapperTranslate(0, scrollPosition, 0);
                swiper.resizeFix(true);
            }
        };

        // return true or false
        this.showClock = function (time) {
            time = moment(time, 'hh:mm:ss');
            return (time.isAfter(dayTimeMin) && time.isBefore(dayTimeMax));

        }

        // returns minute rotation value
        this.minuteRotation = function (time) {
            time = moment(time, 'hh:mm:ss');
            return 6 * time.get('minute');
        }

        // returns hour rotation value
        this.hourRotation = function (time) {
            time = moment(time, 'hh:mm:ss');
            return 0.5 * time.get('minute') + 30 * time.get('hour');
        }
    };

    /**
    * Triggers the coverflip animation and initializes the timer list view.
    *
    * @param {string} detailId Id detail view from which the timer was opened
    * @param {string} timerId Id of the gds timer object
    * @param {string} timerType type of the channel
    */
    module.open = function (detailId, timerId, timerType) {
        currentTimerId = timerId;
        timerTypes = timerType;
        if ($('#timer-view').length !== 0) return;
        loadView('templates/timer-blueprint.xhtml',
            function (data) {
                $(detailId).find('.coverflip-viewsides').append(data);
                if (timerType == "rtc") {
                    $(detailId + ' #timer-view').data('timerActions', { 1: I18nTexts.timerTemplatesRtcTimerActionKomfort, 2: I18nTexts.timerTemplatesRtcTimerActionStandby, 3: I18nTexts.timerTemplatesRtcTimerActionNight, 4: I18nTexts.timerTemplatesRtcTimerActionFrost });
                    $(detailId + ' #rtcTimerAction').show();
                } else {
                    $(detailId + ' #rtcTimerAction').remove();
                }

                if (timerType == "switch") {
                    $(detailId + ' #timer-view').data('timerActions', { 1: I18nTexts.timerTemplatesSwitchTimerActionOn, 2: I18nTexts.timerTemplatesSwitchTimerActionOff });
                    $(detailId + ' #switchTimerAction').show();
                } else {
                    $(detailId + ' #switchTimerAction').remove();
                }

                if (timerType == "dimmer") {
                    $(detailId + ' #timer-view').data('timerActions', { 1: '0 %', 2: '10 %', 3: '20 %', 4: '30 %', 5: '40 %', 6: '50 %', 7: '60 %', 8: '70 %', 9: '80 %', 10: '90 %', 11: '100 %' });
                    $(detailId + ' #dimmerTimerAction').show();
                } else {
                    $(detailId + ' #dimmerTimerAction').remove();
                }

                if (timerType == "blindShutter") {
                    $(detailId + ' #timer-view').data('timerActions', { 1: '0 % ' + I18nTexts.timerTemplatesBlindShutterTimerActionRowText, 2: '10 % ' + I18nTexts.timerTemplatesBlindShutterTimerActionRowText, 3: '20 % ' + I18nTexts.timerTemplatesBlindShutterTimerActionRowText, 4: '30 % ' + I18nTexts.timerTemplatesBlindShutterTimerActionRowText, 5: '40 % ' + I18nTexts.timerTemplatesBlindShutterTimerActionRowText, 6: '50 % ' + I18nTexts.timerTemplatesBlindShutterTimerActionRowText, 7: '60 % ' + I18nTexts.timerTemplatesBlindShutterTimerActionRowText, 8: '70 % ' + I18nTexts.timerTemplatesBlindShutterTimerActionRowText, 9: '80 % ' + I18nTexts.timerTemplatesBlindShutterTimerActionRowText, 10: '90 % ' + I18nTexts.timerTemplatesBlindShutterTimerActionRowText, 11: '100 % ' + I18nTexts.timerTemplatesBlindShutterTimerActionRowText });
                    $(detailId + ' #blindShutterTimerAction').show();
                } else {
                    $(detailId + ' #blindShutterTimerAction').remove();
                }

                if (timerType == "scene") {
                    $(detailId + ' #timer-view').data('timerActions', { 1: I18nTexts.timerTemplatesSceneTimerActionRowText });
                    $(detailId + ' #sceneTimerAction').show();
                } else {
                    $(detailId + ' #sceneTimerAction').remove();
                }

                if (timerType == "blindShutterRelative") {
                    $(detailId + ' #timer-view').data('timerActions', { 1: I18nTexts.timerTemplatesShutterRelativeTimerActionTo, 2: I18nTexts.timerTemplatesShutterRelativeTimerActionFrom});
                    $(detailId + ' #shutterRelativeTimerAction').show();
                } else {
                    $(detailId + ' #shutterRelativeTimerAction').remove();
                }

                timerMode = timerModes.list;
                timerActionNames = $('#timer-view').data('timerActions');
                timerListModel.timers(tsxGdsUtility$.getTimers(currentTimerId));
                ko.applyBindings(timerListModel, $('#timer-view')[0]);
                initToggleButton();
                setTimeout(function() {
                    coverflipFrontToBack(detailId, true);
                    //prevent being stuck in down state
                    $(detailId + ' .timer-icon').trigger('touchcancel');
                }, 0);
                registerTimerListControlButtons();
                registerTimerEditControlButtons();
                registerTimerActionRows();
                registerDaySelectionButtons();
                updateAddButton();
            });
    };

    /**
     * Updates the current timer list and all timer buttons
     *
     * @param {string} id Id of the gds timer object
     * @param {object} object gds timer object
     */
    module.update = function (id, object) {
        if (parseInt(id) === currentTimerId) {
            timerListModel.updateModel();
            disableTimerList(object.enabled === 'true');
            updateAddButton();
            updateEditButton();
            updateDeleteButton();
        }
        var state = object.enabled === 'true' ? globals.timer.buttonActive : globals.timer.button;
        timerIcons[id].forEach(function (element) {
            changeSpriteImage(element, state);
        });
    };

    /**
     * Registers events and casesets for the timer button in a detail view.
     *
     * @param {string} detailId Id of the comprising detail element
     * @param {string} timerId Id of the gds timer object
     * @param {string} timerType type of the channel
     */
    module.registerTimerButton = function(detailId, timerId, timerType) {
        var timerButton = $('#' + detailId + ' .timer-icon');
        registerEventsForSwipeableElement(timerButton,
            function() {
                timerButton.addClass(globals.timer.buttonDown);
            },
            function() {
                timerButton.removeClass(globals.timer.buttonDown);
            },
            function() {
                module.open('#' + detailId, timerId, timerType);
            });
        if (timerIcons[timerId] === undefined) {
            timerIcons[timerId] = [];
        }
        timerIcons[timerId].push(timerButton[0]);
    };

    /**
     * Initializes the timer toggle button with the current value of the gds configuration.
     */
    function initToggleButton() {
        disableTimerList(tsxGdsUtility$.getTimerState(currentTimerId));
        var toggleButton = $('#onoffswitch-timer');
        toggleButton.prop('checked', isTimerEnabled);
        var toggleRow = $('.timer-toggle');
        registerEventsForHorizontalSwipeableElement($('.timer-toggle .onoffswitch'),
            function () {
                if (!toggleRow.hasClass('disabled')) toggleRow.addClass('tile-control-down');
            },
            function () {
                if (!toggleRow.hasClass('disabled')) toggleRow.removeClass('tile-control-down');
            },
            function () {
                toggleButton.prop("checked", !toggleButton.prop("checked")).trigger('change');
            }

        );
        toggleButton.change(function () {
            var enabled = $(this).prop('checked');
            disableTimerList(enabled);
            tsxGdsUtility$.setTimerState(currentTimerId, enabled);
        });
    }

    /**
     * Triggers the ok action depending on the current state of the timer view.
     */
    function ok() {
        var navbarController = NavbarController.getInstance();
        switch (timerMode) {
            case timerModes.list:
                //setEditButtonMode(false);
                timerMode = timerModes.closed;
                navbarController.back();
                break;
            case timerModes.edit:
                setEditButtonMode(false);
                updateEditButton();
                setTimerActiveToggleEnabled(true);
                $('#timer-list-view  .timer-arrow').removeClass('timerFadeOut').addClass('timerFadeIn');
                $('#timer-list-view  .checkbox').removeClass('timerFadeIn edit-mode').addClass('timerFadeOut');
                $('#timer-list-view  .timer-box').removeClass('timerSlideIn edit-mode').addClass('timerSlideOut').one('webkitAnimationEnd',
					function (e) { $('#timer-list-view  .timer-box').removeClass('timerSlideOut'); });
                timerMode = timerModes.list;
                break;
            case timerModes.addTimer:
            case timerModes.editTimer:
                timerMode = previousMode;
                setTimer();
                navbarController.back();
                break;
        }
    };

    function setTimerActiveToggleEnabled(isEnabled) {
        var toggleButton;
        if (isEnabled === true) {
            $('.timer-toggle').removeClass('disabled');
            $('.timer-toggle input').removeAttr("disabled");
            toggleButton = $('.timer-toggle-button');
            toggleButton.removeClass('timerFadeOut').addClass('timerFadeIn');
            toggleButton.css({ "pointer-events": "auto" });

        } else {
            $('.timer-toggle').addClass('disabled');
            $('.timer-toggle input').attr("disabled", true);
            toggleButton = $('.timer-toggle-button');
            toggleButton.removeClass('timerFadeIn').addClass('timerFadeOut');
            toggleButton.css({ "pointer-events": "none" });

        }
    }

    /**
     * Sets the timer list view to the edit mode.
     */
    function edit() {
        setEditButtonMode(true);
        setTimerActiveToggleEnabled(false);
        $('#timer-list-view  .timer-arrow').removeClass('timerFadeIn').addClass('timerFadeOut');
        $('#timer-list-view  .timer-box').removeClass('timerSlideOut').addClass('timerSlideIn');
        $('#timer-list-view  .checkbox').removeClass('timerFadeOut').addClass('timerFadeIn').attr('checked', null);

        updateDeleteButton();
        timerMode = timerModes.edit;
    };

    function setEditButtonMode(deleteModeActive) {
        var editButton = $('#timer-list-view  .timer-controls #edit');
        var deleteButton = $('#timer-list-view  .timer-controls #delete');

        setButtonState(deleteButton, deleteModeActive);
        setButtonState(editButton, !deleteModeActive);

    }

    function setButtonState(button, isVisibleAndClickable) {
        if (isVisibleAndClickable === true) {
            button.css('opacity', '1.0');
            button.css('pointer-events', 'auto');
        } else {
            button.css('opacity', '0.0');
            button.css('pointer-events', 'none');
        }
    }

    /**
     * Closes the edit timer view and restores the previous state.
     */
    function cancel() {
        timerMode = previousMode;
        NavbarController.getInstance().back();
    };

    /**
     * Opens the timer edit view and sets the according initial values for a new timer.
     */
    function addTimer() {
        previousMode = timerMode;
        timerMode = timerModes.addTimer;
        showEditTimerView(undefined, $('#timer-list-view'));
    };

    /**
     * Sets the current state to edit and triggers timer edit view loading.
     */
    function editTimer(timer) {
        previousMode = timerMode;
        timerMode = timerModes.editTimer;
        showEditTimerView(timer, $('#timer-list-view'));
    };

    /**
     * Inititalizes and slides in the timer edit view.
     *
     * @param {boolean} disabled true if the timers are disabled
     */
    function showEditTimerView(timer) {

        //setTimerActiveToggleEnabled(true);
        var src = $('#timer-list-view');
        var target = $('#timer-edit-view');
        var timeWheel = SpinningWheel.getById('timer-time-wheel');

        $('#timer-edit-view').data('settingsSwiper').setWrapperTranslate(0, 0, 0);

        // if existing timer is openend
        if (timer) {
            target.data("timerIndex", timer.index);
            var dayText = timerListModel.formatDays(timer.recurrence, timer.weekdays);
            var timerTime = moment(timer.time, 'HH:mm:ss');

            $('.extended-timer-description', target).text(dayText + ' ' + timerTime.format('HH:mm'));

            $('#timer-view .day-selection-button').each(function (index) {
                if (timer.weekdays !== undefined && timer.recurrence === 'daily' && timer.weekdays[index] === '-') {
                    $(this).removeClass('active');
                } else {
                    $(this).addClass('active');
                }
            });

            timeWheel.scrollToValue(0, timerTime.hours());
            timeWheel.scrollToValue(1, timerTime.minutes());

            $('#timer-view input[name=timerAction]').prop('checked', false);
            var initialAction = $('#timer-view input[name=timerAction][value=' + timer.triggerValue + ']');
            if (initialAction.length === 0) {
                initialAction = $('#timer-view input[name=timerAction]').first();
            }
            initialAction.prop('checked', true);
        } else {
            // if new timer is openend
            target.removeData("timerIndex");
            $('.extended-timer-description', target).text('');
            $('#timer-view .day-selection-button').each(function () {
                $(this).addClass('active');
            });
            timeWheel.scrollToValue(0, '00');
            timeWheel.scrollToValue(1, '00');
            $('#timer-view input[name=timerAction]').first().prop('checked', true);
        }

        var targetCssLeft = parseInt(target.css('left'));
        scrollPosition = $('#timer-list-view').data('settingsSwiper').getWrapperTranslate('y');

        NavbarController.getInstance().addBackFunction(function () {
            timerMode = previousMode;
            src.animate({ left: 0 }, 500, function () {
            });
            target.animate({ left: targetCssLeft }, 500, function () {
                target.hide();
            });
            $('#timer-list-view').data('settingsSwiper').setWrapperTranslate(0, scrollPosition, 0);
        });

        src.animate({ left: -targetCssLeft }, 500, function () {
        });

        target.show().animate({ left: 0 }, 500, function () {
            $('#timer-edit-view').data('settingsSwiper').resizeFix(true);
        });

    };

    /**
     * Enables/disables the add timer button according to the number of elements in the current timer list.
     */
    function updateAddButton() {
        var maxLength = 10;
        if (timerTypes == 'rtc' || timerTypes == '') {
            maxLength = 28;
        }

        var addTimerButton = $('#timer-list-view .add-timer');
        var addTimerButtonImage = $('#timer-list-view .add-timer .sprites');
        if (timerListModel.timers().length < maxLength) {
            addTimerButton.removeClass('disabled');
            addTimerButton.css({ "pointer-events": "auto" });
            spriteEnable(addTimerButtonImage[0]);
        } else {
            addTimerButton.addClass('disabled');
            addTimerButton.css({ "pointer-events": "none" });
            spriteDisable(addTimerButtonImage[0]);
        }
    }

    /**
 * Enables/disables the edit button and "all active" button according to the number of elements in the current timer list.
 */
    function updateEditButton() {
        var editButton = $('#timer-list-view  .timer-controls #edit');
        var toggleRow = $('.timer-toggle');
        var descriptionText = $('#timer-list-view .settings-description-text-item');
        if (timerListModel.timers().length > 0) {
            editButton.removeClass('inactive');
            editButton.css("pointer-events", "auto");
            toggleRow.css('opacity', '1');
            descriptionText.css("display", "none");

        } else {
            editButton.addClass('inactive');
            editButton.css("pointer-events", "none");
            toggleRow.css('opacity', '0');
            descriptionText.css("display", "block");
        }
    }

    /**
     * Reads all values from the current timer edit view and writes a new timer to the gds configuration
     * only daily timers available at the moment!!!
     */
    function setTimer() {
        var newTimer = {};
        var timerIndex = $('#timer-edit-view').data('timerIndex');
        if (timerIndex) {
            newTimer.index = timerIndex;
        } else {
            newTimer.index = timerListModel.timers().length + 1;
        }
        var timeWheelValues = SpinningWheel.getById('timer-time-wheel').getSelectedValues();
        newTimer.time = timeWheelValues[0].value + ':' + timeWheelValues[1].value + ':00';
        newTimer.recurrence = 'daily';
        newTimer.weekdays = '';
        newTimer.timeType = "pointInTime",
        $('#timer-view .day-selection-button').each(function (index) {
            if (!$(this).hasClass('active')) {
                index = "-";
            }
            newTimer.weekdays = newTimer.weekdays.concat(index);
        });
        newTimer.triggerValue = $('#timer-view input[name=timerAction]:radio:checked').val();

        //deactivate if no weekdays are selected
        if (newTimer.weekdays !== '-------')
            newTimer.active = true;
        else
            newTimer.active = false;

        var timerCount = tsxGdsUtility$.getTimers(currentTimerId).length;
        tsxGdsUtility$.setTimer(currentTimerId, newTimer);
        if (timerCount == 0) {
            var toggleButton = $('#onoffswitch-timer');
            toggleButton.prop('checked', true);
            tsxGdsUtility$.setTimerState(currentTimerId, true);
            disableTimerList(true);
        }
    };



    /**
    * Disables/enables timer row elements
    *
    * @param {boolean} disabled true if the timers are disabled
    */
    function disableTimerList(enabled) {
        isTimerEnabled = enabled;
        var timerRows = $('#timer-list-view .timer-row');
        var timerClocksSprites = $('.timer-clock.sprites, .timer-clock .sprites', timerRows);

        var arrows = $('.timer-arrow', timerRows);
        if (!enabled) {
            timerClocksSprites.each(function () {
                spriteDisable(this);
            });
            timerRows.addClass('disabled');
            arrows.hide();
        } else {
            timerRows.removeClass('disabled');
            if (timerMode === timerModes.list) {
                arrows.show();
                timerClocksSprites.each(function () {
                    spriteEnable(this);
                });
            }
        }
    };

    /**
     * update delete button if timers are available activate, if not deactivate
     */
    function updateDeleteButton() {
        var deleteButton = $('#timer-list-view .timer-controls #delete');
        var checkedTimers = $('#timer-list-view .checkbox[checked="checked"]');
        if (checkedTimers.length > 0) {
            deleteButton.removeClass('inactive');
        } else {
            deleteButton.addClass('inactive');
        }
    }

    /**
     * delete choosen timer
     */
    function del() {
        var checkedTimers = $('#timer-list-view .checkbox[checked="checked"]');
        var timerDeleteList = [];
        checkedTimers.each(function (index, element) {
            timerDeleteList.push(parseInt(element.getAttribute('timerId')));
        });
        var timerCount = tsxGdsUtility$.getTimers(currentTimerId).length;
        tsxGdsUtility$.deleteTimers(currentTimerId, timerDeleteList);
        if (timerCount == timerDeleteList.length) {
            var toggleButton = $('#onoffswitch-timer');
            toggleButton.prop('checked', false);
            tsxGdsUtility$.setTimerState(currentTimerId, false);
            disableTimerList(false);
        }
        scrollPosition = $('#timer-list-view').data('settingsSwiper').getWrapperTranslate('y');
        updateDeleteButton();
    }

    /*
     * changes classes when the settingsEntry is touchstart or mousedown
     * to change the colors and images
     *
     * @param {string} settingsEntry the name of the selected settings entry
     * @param {Event} event touchstart or mousedown event
     */
    function onTouchStart(element, event) {
        if ($(event.target).hasClass("ignoreRowEvent"))
            return;
        if (!element.hasClass('down') && !element.hasClass('inactive')) {
            element.addClass('down');
            element.find('.flow-top').css('visibility', 'visible');
            element.find('.timer-clock').addClass('down');

            element.next('.timer-action-gridline').first().css('visibility', 'hidden');

            var sprites = element.find('.sprites:not(.noDownEffect)');
            $(sprites).each(function (i, sprite) {
                spriteDown(sprites[i]);
            });
        }
    };

    /*
     * changes classes when the settingsEntry is touchend or mouseup
     * to change the colors and images
     *
     * @param {string} settingsEntry
     *			the name of the selected settings entry
     */
    function onTouchEnd(settingsEntry) {
        if (settingsEntry.hasClass('down')) {

            settingsEntry.next('.timer-action-gridline').first().css('visibility', 'visible');

            var sprites = settingsEntry.find('.sprites:not(.noDownEffect)');
            $(sprites).each(function (i, sprite) {
                spriteUp(sprites[i]);
            });

            settingsEntry.removeClass('down');
            settingsEntry.find('.timer-clock').removeClass('down');
            settingsEntry.find('.flow-top').css('visibility', 'hidden');

        }
    };

    /**
    * Registers events for timer action rows of the timer list view.
    */
    function registerTimerActionRows() {
        $("#timer-view .timer-action-row").each(function (index, element) {
            var $element = $(element);
            registerEventsForHorizontalSwipeableElement($element,
                function (event) {
                    onTouchStart($element, event);
                },
                function () {
                    onTouchEnd($element);
                },
                function () {
                    $('.radio-input', $element).prop('checked', true);
                }
            );
        });
    }


    /**
     * Registers events for edit, delete, ok and add buttons of the timer list view.
     */
    function registerTimerListControlButtons() {
        var editBtn = $('#timer-list-view .timer-controls #edit');
        var okBtn = $('#timer-list-view .timer-controls #ok');
        var delBtn = $('#timer-list-view .timer-controls #delete');
        var addBtn = $('#timer-list-view .timer-controls .add-timer');
        updateEditButton();

        registerEventsForPressButton(editBtn,
            function () { editBtn.addClass('down'); },
            function () { editBtn.removeClass('down'); },
            function () { edit(); }
        );

        registerEventsForPressButton(delBtn,
            function () { if (!delBtn.hasClass('inactive')) delBtn.addClass('down'); },
            function () { if (!delBtn.hasClass('inactive')) delBtn.removeClass('down'); },
            function () { if (!delBtn.hasClass('inactive')) del(); }
        );
        registerEventsForPressButton(okBtn,
            function () { okBtn.addClass('down'); },
            function () { okBtn.removeClass('down'); },
            function () { ok(); }
        );

        registerEventsForPressButton(addBtn,
            function () {
                if (!addBtn.hasClass('disabled')) addBtn.addClass('down');
            },
            function () {
                if (!addBtn.hasClass('disabled')) addBtn.removeClass('down');
            },
            function () {
                if (!addBtn.hasClass('disabled')) {
                    addBtn.addClass('disabled');
                    addTimer();
                    setTimeout(function () {
                        addBtn.removeClass('disabled')
                    }, 1000);
                }
            }
        );
    }

    /**
     * Registers events for cancel and ok buttons of the timer edit view.
     */
    function registerTimerEditControlButtons() {
        var cancelBtn = $('#timer-edit-view .timer-controls .left');
        var okBtn = $('#timer-edit-view .timer-controls .right');


        registerEventsForPressButton(cancelBtn,
            function () { cancelBtn.addClass('down'); },
            function () { cancelBtn.removeClass('down'); },
            function () {
                cancel();
            }
        );

        registerEventsForPressButton(okBtn,
            function () { okBtn.addClass('down'); },
            function () { okBtn.removeClass('down'); },
            function () {
                ok();
            }
        );
    }

    /**
     * Registers events for the day selection buttons in the timer edit view.
     */
    function registerDaySelectionButtons() {
        $(".day-selection-button").each(function (index, element) {
            registerEventsForHorizontalSwipeableElement($(element), undefined, undefined, function () {
                $(element).toggleClass('active');
            });
        });
    }

    /**
    * Registers events and casesets for the timer button in a detail view.
    *
    * @param {string} detailId Id of the comprising detail element
    * @param {string} timerType type of the channel
    * @param {string} timerId Id of the gds timer object
    */
    module.registerTimer = function (detailId, timerType, channelOffset, timerFunctions) {
        var timerId = parseInt(globals.timer.rfcClockStartId + (parseInt(channelOffset) * globals.timer.rfcClockOffset));
        tsxGdsUtility$.registerTimer(timerId, timerFunctions);
        module.registerTimerButton(detailId, timerId, timerType);
    }

    return module;

})();

