/**
 * Handles the connection with the gdsApi
 *  
 * @param {string} url 
 *          The WebSocket-Server's url
 * @param {boolean} isProxy 
 *          true if url is websocket proxy
 */
window.tsx.GdsConnector = (function (url, isProxy) {
    var m = {};
    var _callback = null;
    var pingTimeoutId;
    var pingIntervalId;
    gds$.registerSocketEventHandlers({
        open: function() {
            fireEvent(_callback.connectionEvents.CONNECTED);
        },
        close: function() {
            fireEvent(_callback.connectionEvents.DISCONNECTED);
        }
    });
    gds$.addValueHandler(processResponse.bind(this));

    /**
     * Connects via WebSocket to HomeServer and registers event handlers
     *  
     * @param {string} host
     *          host name
     * @param {string} port
     *          port of host
     * @param {string} username
     *          username for login
     * @param {string} password
     *          password for login
     */
    m.connect = function (host, port, username, password) {
        fireEvent(_callback.connectionEvents.CONNECTING);
        host = host ? host : 'localhost';
        gds$.connect("ws://" + host + ":8182/gds/api");
    };

    /**
     * Disconnects from HomeServer/WebSocket
     */
    m.disconnect = function () {
        fireEvent(_callback.connectionEvents.DISCONNECTING);
        gds$.close();
    };

    /**
     * Convenience function for sending comands to Homeserver
     * 
     * @param {type} command to send
     * @returns {undefined}
     */
    m.sendCommand = function (command) {
        gds$.setValue(command.tag, command.value);
    };

    /**
     * Implements main state machine for connection handling
     * 
     * @param {string} response 
     * @private
     */
    function processResponse(event) {
        _callback.onTagValue(event.value.id, event.value.new, true);
    };

    /**
     * Sets callback
     * 
     * @param {type} callback
     */
    m.setCallback = function (callback) {
        _callback = callback;
    };

    /**
     * Calls the fireEvent function of _callback if it is not null
     * 
     * @param event 
     *          connection event
     * @param message 
     *          optional event 
     */
    function fireEvent(event, message) {
        if (_callback !== null) {
            _callback.fireEvent(event, message);
        } else {
            console.debug('Event ' + event + '(' + message + ') was not fired because ComClient instance is not available(_callback == null)');
        }
    };

    return m;
});