local json = require("json")

--[[
valid operation modes:
"knx": knx system, configured via ETS project
"visu-client": visu client for other devices
"homeserver-enet": android version with homeserver or enet client
"doorcomm-weather": standalone with only doorcom and weather
"unknown": unknown
]]
local operationMode = nil

function GetVersion()
  return 3
end

function GetCommands()
  return json.encode(
    {
      {
        type = "function",
        func = "Init"
      },
      {
        type = "api",
        data = json.encode({command = "getdeviceconfig", ipc = true}),
        func = "HandleGetDeviceConfig"
      },
      {
        type = "api",
        data = json.encode({command = "getappvalue", appname = "gira-app", key = "first-installation-configuration"}),
        func = "HandleGetAppData"
      },
      {
        type = "api",
        data = json.encode({command = "getconfiguration", object = {urn = "urn:gds:chn:GIG1LXKXIP:Temperature"}}),
        func = "HandleGetConfigTemperaturChannel"
      },
      {
        type = "function",
        func = "GetG1Config"
      },
      {
        type = "api",
        data = json.encode({command = "GetSystemInfo"}),
        func = "HandleGetSystemInfo"
      }
    }
  )
end

function Init()
  -- reset operation mode
  operationMode = nil
  return "{}"
end

function HandleGetConfigTemperaturChannel(response)
  local j = json.decode(response)
  local parameters = GetFromTable(j, "response", "object", "parameters")

  local roomTempUse = false
  local temperaturSensorMode = "unknown"
  if parameters
  then
    for i, parameter in ipairs(parameters) do
      local parameterSet = GetFromTable(parameter, "set")
      local parameterKey = GetFromTable(parameter, "key")
      local parameterValue = GetFromTable(parameter, "value")
      if (parameterSet and parameterKey and parameterValue)
      then
        if (parameterSet == "RoomTemperatureDetection" and parameterKey == "RoomTempUse" and parameterValue:upper() == "TRUE")
        then
          roomTempUse = true
        elseif (parameterSet == "RoomTemperatureDetection" and parameterKey == "RoomTempSensorType")
        then
          if (parameterValue == "0")
          then
            temperaturSensorMode = "internal"
          elseif (parameterValue == "1")
          then
            temperaturSensorMode = "external"
          elseif (parameterValue == "2")
          then
            temperaturSensorMode = "combined"
          end
        end
      end
    end
  end
  return json.encode({config = {temperaturSensor = roomTempUse and temperaturSensorMode or "unknown"}})
end

function HandleGetDeviceConfig(response)
  local j = json.decode(response)
  local devcfg = GetFromTable(j, "response", "deviceConfig", "ipc")
  if devcfg
  then
    local fwv = GetFromTable(devcfg, "CurrentFirmwareVersion")
    local manu = GetFromTable(devcfg, "ManufacturerId")
    local device = GetFromTable(devcfg, "DeviceId")
    local revision = GetFromTable(devcfg, "ManufacturerRevision")
    if (revision == nil or revision == "")
    then
      revision = "unknown"
    end
    if (GetFromTable(devcfg, "CurrentPlatform") == "android")
    then
      operationMode = "homeserver-enet"
    end
    return json.encode({firmwareVersion = fwv, model = (manu .. device), manufacturerRevision = revision})
  end
  return "{}"
end

function HandleGetAppData(response)
  local j = json.decode(response)
  local appdata = GetFromTable(j, "response", "value")

  if appdata
  then
    local firstInstallConfig = json.decode(appdata)
    operationMode = GetFromTable(firstInstallConfig, "mode")
    -- only get the operation mode from first install config, do the rest afterwards
  end

  return "{}"
end

function GetG1Config()
  local weather = nil

  if (operationMode == nil)
  then
    -- if etsproject.xml exists, G1 is in knx mode
    if (CheckIfFileExists("/opt/userdata/etsproject/etsproject.xml"))
    then
      operationMode = "knx"
    end
  end

  -- check the local storage to get operation mode if not yet set and to get weather info for visu-client mode
  if (operationMode == nil or operationMode == "visu-client")
  then
    local localstorage = io.open("/opt/userdata/.chromium/Local Storage/file__0.localstorage", "rb")
    if localstorage
    then
      localstorage:seek("set", 100) -- first 100 bytes are header
      local data = localstorage:read("*all")
      localstorage:close()

      if (operationMode == nil)
      then
        if (DataContains(data, "standaloneDcstrue"))
        then
          operationMode = "doorcomm-weather"
        elseif (DataContains(data, "g1StartupModeclient"))
        then
          operationMode = "visu-client"
        elseif (DataContains(data, "g1StartupModeregular"))
        then
          operationMode = "knx"
        end

        operationMode = operationMode or "unknown"
      end

      -- in operation mode visu-client the weather config is stored on the remote device, so check it from data base
      if (operationMode == "visu-client")
      then
        weather = DataContains(data, "includeWeathertrue") and "yes" or "no"
      end
    end
  end

  -- check if weather is used
  if (weather == nil)
  then
    local weatherSettings = io.open("/opt/userdata/devicestack/appData/Gira.G1/weather.settings", "r")
    if weatherSettings
    then
      local data = weatherSettings:read("*all")
      weatherSettings:close()

      local pos1, pos2 = data:find("\"weatherStations\":%[")
      if (pos1 ~= nil and pos2 ~= nil and data:sub(pos2 + 1, pos2 + 1) ~= "]")
      then
        weather = "yes"
      end
    end

    weather = weather or "no"
  end

  -- check homeserver and enet apps on android
  local homeserver = "unknown"
  local enet = "unknown"
  if (operationMode == "homeserver-enet")
  then
    -- homeserver is running if process de.gira.homeserver.android exists
    -- enet is running if process de.insta.enet.smarthome exists
    local handle = io.popen("/bin/ps")
    if handle
    then
      local data = handle:read("*all")
      handle:close()
      homeserver = data:find("de.gira.homeserver.android") and "yes" or "no"
      enet = data:find("de.insta.enet.smarthome") and "yes" or "no"
    end
  end

  return json.encode({config = {mode = operationMode, weather = weather, homeserver = homeserver, enet = enet}})
end

function HandleGetSystemInfo(response)
  local j = json.decode(response)
  local doorComType = "no"

  local clientTable = GetFromTable(j, "response", "system", "basePlatform", "webSocketHandler", "clients")
  if clientTable
  then
    for key, value in pairs(clientTable) do
      local clientId = GetFromTable(value, "clientID")
      if clientId
      then
        if (clientId:find("^tks_ip_gw_proxy::"))
        then
          doorComType = "gira"
          break
        end
        if (clientId:find("^SipClient::"))
        then
          doorComType = "sip"
          break
        end
      end
    end
  end

  return json.encode({config = {doorcomm = doorComType}})
end

function GetFromTable(tbl, key, ...)
  return not key and tbl or tbl[key] and GetFromTable(tbl[key], ...)
end

function CheckIfFileExists(filename)
  local f = io.open(filename, "r")
  if (f ~= nil) then
    f:close()
    return true;
  end
  return false;
end

function DataContains(data, searchString)
  searchString = searchString:gsub("(.)", "%1.")
  return data:find(searchString) ~= nil
end
