#!/bin/sh

print_usage()
{
  echo "Usage:"
  echo "    ./${0} /block/device /path/to/bootloader.file"
}

if [ "$#" -ne 2 ]
then
  echo "Illegal number of parameters!"
  print_usage
  exit 1
fi

BLOCK_DEVICE="${1}"
BOOTLOADER="${2}"

if ! [ -b "${BLOCK_DEVICE}" ]
then
  echo "Device ${BLOCK_DEVICE} does not exist or is not a block device!"
  print_usage
  exit 1
fi

if ! [ -f "${BOOTLOADER}" ]
then
  echo "File ${BOOTLOADER} does not exist or is not a regular file!"
  print_usage
  exit 1
fi

FILESIZE=$(ls -l "${BOOTLOADER}" | awk '{ print $5 }')
# This checksum is only accurate if the new bootloader is the same size as the
# one that's already installed. But since that's the only case we want to check,
# that's enough. Don't mistake it for the actual md5sum of the installed
# bootloader though.
MD5_CURRENT=$(dd if="${BLOCK_DEVICE}" bs=1 count="${FILESIZE}" skip=1024 | md5sum | cut -d " " -f1)
MD5_NEW=$(md5sum "${BOOTLOADER}" | cut -d " " -f1)

if [ "${MD5_CURRENT}" = "${MD5_NEW}" ]
then
  echo "Current bootloader is already installed, do not replace."
else
  echo "Exchanging bootloader..."
  dd if="${BOOTLOADER}" bs=1K seek=1 of="${BLOCK_DEVICE}"
  if [ "$?" -ne 0 ]
  then
    echo "Error replacing the bootloader!"
    exit 1
  else
    echo "Bootloader successfully replaced!"
  fi 
fi
