import {Log} from "./gira.log.js";
import {IDisposable} from "./gira.lifeCycle.js";

export interface Event<T> {
    (listener: (e: T) => any, thisArgs?: any, disposables?: IDisposable[]): IDisposable;
}

export namespace Event {
    const _disposable = { dispose() { } };
    export const None: Event<any> = () => _disposable;
}

type Listener = [Function, any] | Function;

export class EventEmitter<T> {
    private static readonly _noop = () => { };
    private _event: Event<T>;
    private _listeners: {[key: string]: (e: T) => any | any[] | undefined} = {};
    private _deliveryQueue: Array<[Listener, T | undefined]> = [];
    private _disposed: boolean;

    private createDisposableObject(removeKey: number): IDisposable {
        let result: IDisposable;
        result = {
            dispose: () => {
                result.dispose = EventEmitter._noop;
                if (!this._disposed) {
                    delete this._listeners[removeKey];
                }
            }
        };
        return result;
    }

    private attachEvent(listener: (e: T) => any, thisArgs?: any, disposables?: IDisposable[]): IDisposable {
        const key: number = Object.keys(this._listeners).length;
        this._listeners[key] = (!thisArgs ? listener : [listener, thisArgs]) as (e: T) => any | any[];
        let result: IDisposable = this.createDisposableObject(key);
        if (Array.isArray(disposables)) {
            disposables.push(result);
        }
        return result;
    }

    public fire(event?: T): any {
        Object.keys(this._listeners).forEach((key: string) => {
            this._deliveryQueue.push([this._listeners[key], event]);
        });
        while (this._deliveryQueue.length > 0) {
            const [listener, eventForSend] = this._deliveryQueue.shift() as any[];
            try {
                if (typeof listener === 'function') {
                    listener.call(undefined, eventForSend);
                } else {
                    listener[0].call(listener[1], eventForSend);
                }
            } catch (ex) {
                Log.exception(ex);
            }
        }
    }

    public get event(): Event<T> {
        if (!this._event) {
            this._event = (listener: (e: T) => any, thisArgs?: any, disposables?: IDisposable[]) => {
                return this.attachEvent(listener, thisArgs, disposables);
            };
        }
        return this._event;
    }

    public dispose(): void {
        if (!this._disposed) {
            Object.keys(this._listeners).forEach((key: string) => {
                delete this._listeners[key];
            });
            this._deliveryQueue.length = 0;
            this._disposed = true;
        }
    }
}