local json = require("json")
local operationMode = nil

function GetVersion()
  return 1
end

function GetCommands()
  return json.encode(
    {
      {
        type = "function",
        func = "Init"
      },
      {
        type = "api",
        data = json.encode({command = "getdeviceconfig", ipc = true}),
        func = "HandleGetDeviceConfig"
      },
      {
        type = "api",
        data = json.encode({command = "getappvalue", appname = "gira-app", key = "first-installation-configuration"}),
        func = "HandleGetAppData"
      },
      {
        type = "api",
        data = json.encode({command = "GetConfiguration", object = {urn = "urn:gds:chv:Weather"}}),
        func = "HandleWeatherInfo"
      },
      {
        type = "api",
        data = json.encode({command = "GetSystemInfo"}),
        func = "HandleGetSystemInfo"
      }
    }
  )
end

function Init()
  -- wait for GPA project to be loaded
  os.execute("sleep 10")

  -- reset operation mode
  operationMode = nil
  return "{}"
end

function HandleGetDeviceConfig(response)
  local j = json.decode(response)
  local devcfg = GetFromTable(j, "response", "deviceConfig", "ipc")
  if devcfg
  then
    local fwv = GetFromTable(devcfg, "CurrentFirmwareVersion")
    local manu = GetFromTable(devcfg, "ManufacturerId")
    local device = GetFromTable(devcfg, "DeviceId")
    return json.encode({firmwareVersion = fwv, model = (manu .. device)})
  end
  return ""
end

function HandleGetAppData(response)
  local j = json.decode(response)
  local appdata = GetFromTable(j, "response", "value")

  if appdata
  then
    local firstInstallConfig = json.decode(appdata)
    operationMode = GetFromTable(firstInstallConfig, "mode")
    return json.encode({config = firstInstallConfig})
  end

  return json.encode({config = {mode = "unknown", doorcomm = "unknown", weather = "unknown"}})
end

function HandleWeatherInfo(response)
  -- only parse response if operation mode is "knx"
  if operationMode ~= "knx"
  then
    return "{}"
  end

  local j = json.decode(response)
  local weather_chv = GetFromTable(j, "response", "object")
  local weatherConfig = weather_chv and "yes" or "no"

  return json.encode({config = {weather = weatherConfig}})
end

function HandleGetSystemInfo(response)
  -- only parse response if operation mode is "knx"
  if operationMode ~= "knx"
  then
    return "{}"
  end

  local j = json.decode(response)
  local doorComType = "no"

  local clientTable = GetFromTable(j, "response", "system", "basePlatform", "webSocketHandler", "clients")
  if clientTable
  then
    for key, value in pairs(clientTable) do
      local clientId = GetFromTable(value, "clientID")
      if clientId
      then
        if (clientId:find("^tks_ip_gw_proxy::"))
        then
          doorComType = "gira"
          break
        end
        if (clientId:find("^SipClient::"))
        then
          doorComType = "sip"
          break
        end
      end
    end

    return json.encode({config = {doorcomm = doorComType}})
  end

  return ""
end

function GetFromTable(tbl, key, ...)
  return not key and tbl or tbl[key] and GetFromTable(tbl[key], ...)
end
