-- KNX Application Program Handler
-- URNs, IDs and COs in this file are specific to the G1
--
-- Module with specific functions to G1
local g1 = {}

local m = require("knxappprghandler_base")
local DPT = m.DPT

local IdTable = {}
-- de.gira.schema.channels.GdsDevice
IdTable[1002] =  1  -- Ready
IdTable[1003] =  2  -- State
IdTable[1004] =  3  -- Reset
IdTable[1005] =  4  -- Local-Time
IdTable[1007] =  8  -- Uptime
-- de.gira.schema.channels.KnxProgrammingModes
IdTable[1051] = 51  -- Programming-Mode-Device-1
-- de.gira.schema.channels.KnxBusStates
IdTable[1061] = 60  -- Bus-State-Device-1

m.InitIdTable(IdTable)

local CoTable = {}
CoTable[ 1] = { 1002, DPT( 1,  11) }  -- Ready
CoTable[ 2] = { 1003, DPT( 5,  10) }  -- State
CoTable[ 3] = { 1004, DPT( 1,  15) }  -- Reboot
CoTable[ 4] = { 1005, DPT(11,   1) }  -- Date (only as time server)
CoTable[ 5] = { 1005, DPT(10,   1) }  -- Time (only as time server)
CoTable[ 6] = { 1005, DPT(11,   1) }  -- Set Date (only as time client)
CoTable[ 7] = { 1005, DPT(10,   1) }  -- Set Time (only as time client)
CoTable[ 8] = { 1007, DPT(13, 100) }  -- Uptime
CoTable[50] = { 1051, DPT( 1,   1) }  -- Programming Mode
CoTable[51] = { 1051, DPT( 1,   1) }  -- Programming Mode State
CoTable[60] = { 1061, DPT( 1,  11) }  -- Bus State

m.InitCoTable(CoTable)

local device = { firstId = 1002, lastId = 1061, firstCo = 1, lastCo = 60 }
m.InitDeviceTable(device)

local channelTable = {
  {
    name = "GDS-Device-Channel",
    dps = {
      {
        name = "Ready",
        dpt = DPT(1, 11),
        rCo = 1,
        event = true
      },
      {
        name = "State",
        dpt = DPT(5, 10),
        rCo = 2,
        event = true
      },
      {
        name = "Reset",
        dpt = DPT(1, 15),
        wCo = 3
      },
      {
        name = "Local-Time", -- date
        dpt = DPT(11, 1),
        rCo = 4,
        event = false,
        internal = false
      },
      {
        name = "Local-Time", -- time
        dpt = DPT(10, 1),
        rCo = 5,
        event = false,
        internal = false
      },
      {
        name = "Local-Time", -- date
        dpt = DPT(11, 1),
        wCo = 6,
        event = false,
        internal = false
      },
      {
        name = "Local-Time", -- time
        dpt = DPT(10, 1),
        wCo = 7,
        event = false,
        internal = false
      },
      {
        name = "Uptime",
        dpt = DPT(13, 100),
        rCo = 8,
        event = false,
        internal = false
      }
    }
  },

  {
    name = "KNX-Programming-Modes",
    dps = {
      {
        name = "Programming-Mode-Device-1",
        dpt = DPT(1, 1),
        wCo = 50,
        rCo = 51,
        event = true
      }
    }
  },
  {
    name = "Temperature",
    dps = {
      {
        name = "InternalTemperature",
        dpt = DPT(9, 1),
        rCo = 67,
        event = true,
        internal = false
      },
      {
        name = "ExternalTemperature",
        dpt = DPT(9, 1),
        wCo = 66,
        event = false,
        internal = true
      },
      {
        name = "OutdoorTemperature",
        dpt = DPT(9, 1),
        wCo = 68,
        event = false,
        internal = true
      },
      {
        name = "SensorTemperature",
        dpt = DPT(9, 1)
      },
      {
        name = "SensorTemperatureStatus",
        dpt = DPT(5, 10)
      }
    },
  },
  {
    name = "RTC-1",
    dps = {
      {
        name = "Current",
        dpt = DPT(9, 1)
        -- no CO
      },
      {
        name = "Set-Point-Bus",
        dpt = DPT(9, 1),
        wCo = 71,
        rCo = 72,
        event = true,
        internal = false
      },
      {
        name = "Mode",
        dpt = DPT(20, 102),
        wCo = 73,
        internal = false
      },
      {
        name = "Status",
        dpt = DPT(20, 105),
        -- no CO
      },
      {
        name = "Presence",
        dpt = DPT(1, 18),
        wCo = 75,
        rCo = 76,
        event = true,
        internal = false
      },
      {
        name = "Heating",
        dpt = DPT(1, 2),
        rCo = 77,
        event = true,
        internal = false
      },
      {
        name = "Cooling",
        dpt = DPT(1, 2),
        rCo = 78,
        event = true,
        internal = false
      },
      {
        name = "Changeover-Mode",
        dpt = DPT(20, 107),
        wCo = 80,
        rCo = 81,
        event = true,
        internal = false
      },
      {
        name = "Window-Contact",
        dpt = DPT(1, 19),
        wCo = 82,
        rCo = 83,
        event = true,
        internal = false
      },
      {
        name = "Forced-Mode",
        dpt = DPT(20, 102),
        wCo = 84,
        rCo = 85,
        event = true,
        internal = false
      },
      {
        name = "Forced-Status",
        dpt = DPT(20, 105)
        -- no CO
      },
      {
        name = "Heating-Variable",
        dpt = DPT(5, 1),
        rCo = 86,
        event = true,
        internal = false
      },
      {
        name = "Heating-Switch",
        dpt = DPT(1, 1),
        rCo = 87,
        event = true,
        internal = false
      },
      {
        name = "Cooling-Variable",
        dpt = DPT(5, 1),
        rCo = 88,
        event = true,
        internal = false
      },
      {
        name = "Cooling-Switch",
        dpt = DPT(1, 1),
        rCo = 89,
        event = true,
        internal = false
      },
      {
        name = "Locked",
        dpt = DPT(1, 2),
        wCo = 90,
        rCo = 91,
        event = true,
        internal = false
      },
      {
        name = "Limited",
        dpt = DPT(1, 2),
        wCo = 92,
        rCo = 93,
        event = true,
        internal = false
      },
      {
        name = "Disabled",
        dpt = DPT(1, 2),
        wCo = 94,
        rCo = 95,
        event = true,
        internal = false
      },
      {
        name = "Common-Status",
        dpt = DPT(20, 105),
        rCo = 74,
        event = true,
        internal = false
      },
      {
        name = "Extended-Status",
        dpt = DPT(22, 101),
        rCo = 96,
        event = true,
        internal = false
      },
      {
        name = "Heat-Cool",
        dpt = DPT(1, 100),
        wCo = 97,
        rCo = 98,
        event = true,
        internal = false
      }
    }
  },
  {
    name = "FloorCall",
    dps = {
      {
        name = "Trigger",
        dpt = DPT(1, 17),
        wCo = 62,
        internal = false
      }
    },
  }
}

local function file_exists(name)
  local f=io.open(name,"r")
  if f~=nil then
    io.close(f)
    return true
  else
    return false
  end
end

package.path = package.path .. ";/opt/userdata/devicestack/?.lua"

if (file_exists("knxappprghandler_urn_generated.lua") or
    file_exists("/opt/userdata/devicestack/knxappprghandler_urn_generated.lua")) then
  local generatedTable = require("knxappprghandler_urn_generated")

  -- append generated table to existing table
  for i = 1, #generatedTable do
    channelTable[#channelTable + 1] = generatedTable[i]
  end
end

m.InitUrn(channelTable)

local timeCoTable = {
  {
    dpt = DPT(11, 1),
    wCo = 4,
    rCo = 6
  },
  {
    dpt = DPT(10, 1),
    wCo = 5,
    rCo = 7
  }
}

m.InitTimeCoTable(timeCoTable)

--[[
KNX properties management
]]

-- fix value could disable by set to 0 only
-- see KnxStack::InterfaceObjectType or (own type > 50000 and ID > 50)
local InterfaceObjectId = 0x0003
m.InitInterfaceObjectId(InterfaceObjectId)

-- see KNX specification (for Object type [0..99] the ID > 200)
local PropertyTable = {}
-- row = Property ID, type (KnxStack::PropertyDatatype), name
PropertyTable[1] = {201, 0x35, "TimeMode", 1} -- PDT_ENUM8

m.InitPropertyTable(PropertyTable)

local InterfaceObjectIdFunctions = 50001
-- fan coil
local InterfaceObjectIdFC        = 50002
-- RTC
local InterfaceObjectIdRtc       = 50003

local G1FolderCount     = 6
local G1ChannelCount    = 150
-- Extended Data
-- fix value could disable by set to 0 only
-- see KnxStack::InterfaceObjectType or (own type > 50000 and ID > 50)
g1.ExtendedPropertyTable = {}
-- row = Property ID, type (KnxStack::PropertyDatatype), arrayCount(1 = single value)
-- General and generic parameters
g1.ExtendedPropertyTable["MaxFunctionPerFolder"]  = {PropertyId = 51, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //NumFunctions
g1.ExtendedPropertyTable["MaxNameCharCount"]      = {PropertyId = 53, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //NumNameChars
g1.ExtendedPropertyTable["MaxTextCharCount"]      = {PropertyId = 54, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //NumTextChars
g1.ExtendedPropertyTable["MaxUrlCharCount"]       = {PropertyId = 55, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //NumUrlChars

g1.ExtendedPropertyTable["UseGPA"]                = {PropertyId = 60, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["UseDCS"]                = {PropertyId = 61, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["UseWeather"]            = {PropertyId = 62, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["ShowOutdoorTemp"]       = {PropertyId = 63, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["ShowIndoorTemp"]        = {PropertyId = 64, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["UseTimers"]             = {PropertyId = 65, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["UseHeatingTimers"]      = {PropertyId = 66, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["UseDcsFloorcall"]       = {PropertyId = 67, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["DcsFloorCallNameOffset"]= {PropertyId = 68, Type = 0x09, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_LONG //MemoryOffset
g1.ExtendedPropertyTable["ResetUserData"]         = {PropertyId = 69, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["ShowDateTime"]          = {PropertyId = 70, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["DcsFloorCallMelody"]    = {PropertyId = 71, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //DcsFloorCallMelody

-- Folder and function specific parameters
g1.ExtendedPropertyTable["NumberOfFolders"]       = {PropertyId = 80, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //NumFolders
g1.ExtendedPropertyTable["ShowRootFolder"]        = {PropertyId = 81, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["NumberFuncPerFolder"]   = {PropertyId = 82, Type = 0x02, ArrayCount = G1FolderCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //NumFunctions
g1.ExtendedPropertyTable["FolderNameOffset"]      = {PropertyId = 84, Type = 0x09, ArrayCount = G1FolderCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_LONG //MemoryOffset
g1.ExtendedPropertyTable["FolderShowOnHome"]      = {PropertyId = 85, Type = 0x02, ArrayCount = G1FolderCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["FolderIcon"]            = {PropertyId = 86, Type = 0x04, ArrayCount = G1FolderCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_INT //Iconlist

-- Channels
g1.ExtendedPropertyTable["ChannelUse"]            = {PropertyId = 89, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["ChannelType"]           = {PropertyId = 90, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //Channel
g1.ExtendedPropertyTable["FunctionNameOffset"]    = {PropertyId = 92, Type = 0x09, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_LONG //MemoryOffset
g1.ExtendedPropertyTable["FunctionIcon"]          = {PropertyId = 93, Type = 0x04, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_INT //Iconlist
g1.ExtendedPropertyTable["FunctionShowOnHome"]    = {PropertyId = 94, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["FunctionShowTimer"]     = {PropertyId = 95, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Show
-- Switch specific
g1.ExtendedPropertyTable["FunctionSwitchType"]    = {PropertyId = 100, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //Switch
g1.ExtendedPropertyTable["FunctionButtonType"]    = {PropertyId = 101, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //Button
-- Scene specific
g1.ExtendedPropertyTable["FunctionSceneCanLearn"] = {PropertyId = 105, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //CanLearn
g1.ExtendedPropertyTable["FunctionSceneNumber"]   = {PropertyId = 106, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //SceneNumber
-- 'Multiple specific'
g1.ExtendedPropertyTable["FunctionOnColor"]         = {PropertyId = 110, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //Color
g1.ExtendedPropertyTable["FunctionOffColor"]        = {PropertyId = 111, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //Color
g1.ExtendedPropertyTable["FunctionOnTextOffset"]    = {PropertyId = 112, Type = 0x09, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_LONG //MemoryOffset
g1.ExtendedPropertyTable["FunctionOffTextOffset"]   = {PropertyId = 113, Type = 0x09, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_LONG //MemoryOffset
g1.ExtendedPropertyTable["FunctionOnActionOffset"]  = {PropertyId = 114, Type = 0x09, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_LONG //MemoryOffset
g1.ExtendedPropertyTable["FunctionOffActionOffset"] = {PropertyId = 115, Type = 0x09, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_LONG //MemoryOffset
g1.ExtendedPropertyTable["FunctionShowStatus"]      = {PropertyId = 116, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Status
g1.ExtendedPropertyTable["FunctionDefaultShift"]    = {PropertyId = 117, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //DefaultShift
g1.ExtendedPropertyTable["FunctionStepDimTimeout"]  = {PropertyId = 118, Type = 0x0A, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_FLOAT //ButtonTimeout
g1.ExtendedPropertyTable["FunctionStepMoveTimeout"] = {PropertyId = 119, Type = 0x0A, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_FLOAT //ButtonTimeout
g1.ExtendedPropertyTable["FunctionColor"]           = {PropertyId = 120, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //Color
g1.ExtendedPropertyTable["FunctionTextOffset"]      = {PropertyId = 121, Type = 0x09, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_LONG //MemoryOffset
g1.ExtendedPropertyTable["FunctionMinColorTemp"]    = {PropertyId = 122, Type = 0x04, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_INT //ColorTemp
g1.ExtendedPropertyTable["FunctionMaxColorTemp"]    = {PropertyId = 123, Type = 0x04, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_INT //ColorTemp
g1.ExtendedPropertyTable["FunctionSetpointMin"]     = {PropertyId = 124, Type = 0x0A, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_FLOAT //SetpointTemp
g1.ExtendedPropertyTable["FunctionSetpointMax"]     = {PropertyId = 125, Type = 0x0A, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_FLOAT //SetpointTemp
g1.ExtendedPropertyTable["FunctionSetpointShift"]   = {PropertyId = 126, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //SetpointShift
g1.ExtendedPropertyTable["FunctionHVACStatusType"]  = {PropertyId = 127, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //HvacStatusType
-- Value transmitter and status specific
g1.ExtendedPropertyTable["FunctionTransValueType"]    = {PropertyId = 130, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //ValueType
g1.ExtendedPropertyTable["FunctionTransStatusType"]   = {PropertyId = 131, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //StatusType
g1.ExtendedPropertyTable["FunctionTransValueMin100"]  = {PropertyId = 132, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FunctionTransValueMax100"]  = {PropertyId = 133, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FunctionTransValueDef100"]  = {PropertyId = 134, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FunctionTransValueMin255"]  = {PropertyId = 135, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FunctionTransValueMax255"]  = {PropertyId = 136, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FunctionTransValueDef255"]  = {PropertyId = 137, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FunctionTransValueMin128"]  = {PropertyId = 138, Type = 0x01, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_CHAR //FromN128To127
g1.ExtendedPropertyTable["FunctionTransValueMax127"]  = {PropertyId = 139, Type = 0x01, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_CHAR //FromN128To127
g1.ExtendedPropertyTable["FunctionTransValueDef127"]  = {PropertyId = 140, Type = 0x01, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_CHAR //FromN128To127
g1.ExtendedPropertyTable["FunctionTransValueMin99"]   = {PropertyId = 141, Type = 0x0A, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_FLOAT //FromN99To99
g1.ExtendedPropertyTable["FunctionTransValueMax99"]   = {PropertyId = 142, Type = 0x0A, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_FLOAT //FromN99To99
g1.ExtendedPropertyTable["FunctionTransValueDef99"]   = {PropertyId = 143, Type = 0x0A, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_FLOAT //FromN99To99
g1.ExtendedPropertyTable["FunctionTransValueUnit"]    = {PropertyId = 144, Type = 0x24, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_GENERIC_20 //Unit
g1.ExtendedPropertyTable["FunctionTransValueStepsize"]    = {PropertyId = 145, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //StepSize
g1.ExtendedPropertyTable["FunctionTransValueCanChange"]   = {PropertyId = 146, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //CanChange
g1.ExtendedPropertyTable["FunctionTransValueShowCur"]     = {PropertyId = 147, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Show

g1.ExtendedPropertyTable["FunctionTransValueMinU8"]       = {PropertyId = 148, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //RangeU8
g1.ExtendedPropertyTable["FunctionTransValueMaxU8"]       = {PropertyId = 149, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //RangeU8
g1.ExtendedPropertyTable["FunctionTransValueDefU8"]       = {PropertyId = 150, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //RangeU8
g1.ExtendedPropertyTable["FunctionTransValueMinU16"]      = {PropertyId = 151, Type = 0x04, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_INT //RangeU16
g1.ExtendedPropertyTable["FunctionTransValueMaxU16"]      = {PropertyId = 152, Type = 0x04, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_INT //RangeU16
g1.ExtendedPropertyTable["FunctionTransValueDefU16"]      = {PropertyId = 153, Type = 0x04, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_INT //RangeU16
g1.ExtendedPropertyTable["FunctionTransValueMinU32"]      = {PropertyId = 154, Type = 0x09, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_LONG //RangeU32
g1.ExtendedPropertyTable["FunctionTransValueMaxU32"]      = {PropertyId = 155, Type = 0x09, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_LONG //RangeU32
g1.ExtendedPropertyTable["FunctionTransValueDefU32"]      = {PropertyId = 156, Type = 0x09, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_LONG //RangeU32
g1.ExtendedPropertyTable["FunctionTransValueMinS8"]       = {PropertyId = 157, Type = 0x01, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_CHAR //RangeS8
g1.ExtendedPropertyTable["FunctionTransValueMaxS8"]       = {PropertyId = 158, Type = 0x01, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_CHAR //RangeS8
g1.ExtendedPropertyTable["FunctionTransValueDefS8"]       = {PropertyId = 159, Type = 0x01, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_CHAR //RangeS8
g1.ExtendedPropertyTable["FunctionTransValueMinS16"]      = {PropertyId = 160, Type = 0x03, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_INT //RangeS16
g1.ExtendedPropertyTable["FunctionTransValueMaxS16"]      = {PropertyId = 161, Type = 0x03, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_INT //RangeS16
g1.ExtendedPropertyTable["FunctionTransValueDefS16"]      = {PropertyId = 162, Type = 0x03, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_INT //RangeS16
g1.ExtendedPropertyTable["FunctionTransValueMinS32"]      = {PropertyId = 163, Type = 0x08, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_LONG //RangeS32
g1.ExtendedPropertyTable["FunctionTransValueMaxS32"]      = {PropertyId = 164, Type = 0x08, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_LONG //RangeS32
g1.ExtendedPropertyTable["FunctionTransValueDefS32"]      = {PropertyId = 165, Type = 0x08, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_LONG //RangeS32
g1.ExtendedPropertyTable["FunctionTransValueMinD16"]      = {PropertyId = 166, Type = 0x05, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_KNX_FLOAT //RangeD16
g1.ExtendedPropertyTable["FunctionTransValueMaxD16"]      = {PropertyId = 167, Type = 0x05, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_KNX_FLOAT //RangeD16
g1.ExtendedPropertyTable["FunctionTransValueDefD16"]      = {PropertyId = 168, Type = 0x05, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_KNX_FLOAT //RangeD16
g1.ExtendedPropertyTable["FunctionTransValueMinD32"]      = {PropertyId = 169, Type = 0x0A, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_FLOAT //RangeD32
g1.ExtendedPropertyTable["FunctionTransValueMaxD32"]      = {PropertyId = 170, Type = 0x0A, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_FLOAT //RangeD32
g1.ExtendedPropertyTable["FunctionTransValueDefD32"]      = {PropertyId = 171, Type = 0x0A, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_FLOAT //RangeD32
-- status
g1.ExtendedPropertyTable["FunctionTransStatusUnit"]       = {PropertyId = 180, Type = 0x24, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_GENERIC_20 //Unit
g1.ExtendedPropertyTable["FunctionTransStatusDescOffset"] = {PropertyId = 181, Type = 0x09, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_LONG //MemoryOffset
g1.ExtendedPropertyTable["FunctionTransStatusDigits"]     = {PropertyId = 182, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //Digits

-- Trigger specific
g1.ExtendedPropertyTable["FunctionTriggerCommand"]        = {PropertyId = 200, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //TriggerCommand
g1.ExtendedPropertyTable["FunctionTriggerPressAndHold"]   = {PropertyId = 201, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["FunctionTriggerReleaseCommand"] = {PropertyId = 202, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //TriggerCommand

-- IP camera specific
g1.ExtendedPropertyTable["IpCamUrlOffset"]                = {PropertyId = 210, Type = 0x09, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_LONG //MemoryOffset
g1.ExtendedPropertyTable["IpCamCodec"]                    = {PropertyId = 211, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //Codec

-- Sauna heating specific
g1.ExtendedPropertyTable["SaunaHeatingTempRange"]         = {PropertyId = 215, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //SaunaTempRange
g1.ExtendedPropertyTable["SaunaHeatingSetpointShift"]     = {PropertyId = 216, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //SaunaSetpointShift

-- Heating and cooling specific
g1.ExtendedPropertyTable["FunctionHeatCoolMode"]          = {PropertyId = 220, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_ENUM8 //RtcMode
g1.ExtendedPropertyTable["FunctionHeatCoolPresence"]      = {PropertyId = 221, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFunctions} -- PDT_UNSIGNED_CHAR //Show

-- Fan coil specific
g1.ExtendedPropertyTable["FanCoilValueType"]            = {PropertyId = 51, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["FanCoilShowAuto"]             = {PropertyId = 52, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["FanCoilShowHeat"]             = {PropertyId = 53, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["FanCoilShowCool"]             = {PropertyId = 54, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["FanCoilShowFanonly"]          = {PropertyId = 55, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["FanCoilShowDehum"]            = {PropertyId = 56, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["FanCoilUseKnxMode"]           = {PropertyId = 57, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["FanCoilValueModeAuto"]        = {PropertyId = 58, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilValueModeHeat"]        = {PropertyId = 59, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilValueModeCool"]        = {PropertyId = 60, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilValueModeFanOnly"]     = {PropertyId = 61, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilValueModeDehum"]       = {PropertyId = 62, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilShowFanSpeedLevel"]    = {PropertyId = 63, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["FanCoilShowHorizLevel"]       = {PropertyId = 64, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["FanCoilShowHorizStopMove"]    = {PropertyId = 65, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["FanCoilShowVertiLevel"]       = {PropertyId = 66, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["FanCoilShowVertiStopMove"]    = {PropertyId = 67, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["FanCoilNumberFanSpeedLevel"]  = {PropertyId = 68, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_ENUM8 //FanCoilLevel
g1.ExtendedPropertyTable["FanCoilNumberHorizLevel"]     = {PropertyId = 69, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_ENUM8 //FanCoilLevel
g1.ExtendedPropertyTable["FanCoilNumberVertiLevel"]     = {PropertyId = 70, Type = 0x35, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_ENUM8 //FanCoilLevel
g1.ExtendedPropertyTable["FanCoilHorizStopValue01"]     = {PropertyId = 71, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To1
g1.ExtendedPropertyTable["FanCoilHorizMoveValue01"]     = {PropertyId = 72, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To1
g1.ExtendedPropertyTable["FanCoilVertiStopValue01"]     = {PropertyId = 73, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To1
g1.ExtendedPropertyTable["FanCoilVertiMoveValue01"]     = {PropertyId = 74, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To1
g1.ExtendedPropertyTable["FanCoilShowFanSpeedLevelAuto"]= {PropertyId = 75, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["FanCoilShowFanSpeedLevelStop"]= {PropertyId = 76, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["FanCoilShowFanSpeedLevelMax"] = {PropertyId = 77, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["FanCoilFanSpeedLevelAuto"]    = {PropertyId = 78, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilFanSpeedLevelStop"]    = {PropertyId = 79, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilFanSpeedLevelMax"]     = {PropertyId = 80, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilHorizStopLevel"]       = {PropertyId = 81, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To1
g1.ExtendedPropertyTable["FanCoilHorizMoveLevel"]       = {PropertyId = 82, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To1
g1.ExtendedPropertyTable["FanCoilVertiStopLevel"]       = {PropertyId = 83, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To1
g1.ExtendedPropertyTable["FanCoilVertiMoveLevel"]       = {PropertyId = 84, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To1
g1.ExtendedPropertyTable["FanCoilFanSpeedLevel1"]       = {PropertyId = 87, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilFanSpeedLevel2"]       = {PropertyId = 88, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilFanSpeedLevel3"]       = {PropertyId = 89, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilFanSpeedLevel4"]       = {PropertyId = 90, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilFanSpeedLevel5"]       = {PropertyId = 91, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilHorizLevel1"]          = {PropertyId = 92, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilHorizLevel2"]          = {PropertyId = 93, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilHorizLevel3"]          = {PropertyId = 94, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilHorizLevel4"]          = {PropertyId = 95, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilHorizLevel5"]          = {PropertyId = 96, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilVertiLevel1"]          = {PropertyId = 97, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilVertiLevel2"]          = {PropertyId = 98, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilVertiLevel3"]          = {PropertyId = 99, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilVertiLevel4"]          = {PropertyId = 100, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilVertiLevel5"]          = {PropertyId = 101, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To255
g1.ExtendedPropertyTable["FanCoilFanSpeedValueAuto"]    = {PropertyId = 102, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilFanSpeedValueStop"]    = {PropertyId = 103, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilFanSpeedValueMax"]     = {PropertyId = 104, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilHorizStopValue"]       = {PropertyId = 105, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilHorizMoveValue"]       = {PropertyId = 106, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilVertiStopValue"]       = {PropertyId = 107, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilVertiMoveValue"]       = {PropertyId = 108, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilFanSpeedValue1"]       = {PropertyId = 109, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilFanSpeedValue2"]       = {PropertyId = 110, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilFanSpeedValue3"]       = {PropertyId = 111, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilFanSpeedValue4"]       = {PropertyId = 112, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilFanSpeedValue5"]       = {PropertyId = 113, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilHorizValue1"]          = {PropertyId = 114, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilHorizValue2"]          = {PropertyId = 115, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilHorizValue3"]          = {PropertyId = 116, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilHorizValue4"]          = {PropertyId = 117, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilHorizValue5"]          = {PropertyId = 118, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilVertiValue1"]          = {PropertyId = 119, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilVertiValue2"]          = {PropertyId = 120, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilVertiValue3"]          = {PropertyId = 121, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilVertiValue4"]          = {PropertyId = 122, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100
g1.ExtendedPropertyTable["FanCoilVertiValue5"]          = {PropertyId = 123, Type = 0x02, ArrayCount = G1ChannelCount, InterfaceObjectId = InterfaceObjectIdFC} -- PDT_UNSIGNED_CHAR //From0To100

-- row = Property ID, type (KnxStack::PropertyDatatype), arrayCount(1 = single value)
-- Room temperature related
g1.ExtendedPropertyTable["RoomTempUse"]             = {PropertyId = 51, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["RoomTempUnitUse"]         = {PropertyId = 52, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //Show
g1.ExtendedPropertyTable["RoomTempUnitText"]        = {PropertyId = 53, Type = 0x24, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_GENERIC_20 //Unit
g1.ExtendedPropertyTable["RoomTempSensorType"]      = {PropertyId = 54, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RoomTempSensorType
g1.ExtendedPropertyTable["RoomTempSensorRatio"]     = {PropertyId = 55, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RoomTempSensorRatio
g1.ExtendedPropertyTable["RoomTempDelta"]           = {PropertyId = 56, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //RoomTempDelta
g1.ExtendedPropertyTable["RoomTempReadInterval"]    = {PropertyId = 57, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //RoomTempInterval
g1.ExtendedPropertyTable["RoomTempWriteDelta"]      = {PropertyId = 58, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //RoomTempWriteDelta
g1.ExtendedPropertyTable["RoomTempWriteInterval"]   = {PropertyId = 59, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //RoomTempInterval

-- RTC specific parameter
g1.ExtendedPropertyTable["RtcUse"]                  = {PropertyId = 70, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["RtcNameOffset"]           = {PropertyId = 71, Type = 0x09, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_LONG //MemoryOffset
g1.ExtendedPropertyTable["RtcIcon"]                 = {PropertyId = 72, Type = 0x04, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_INT //Iconlist
g1.ExtendedPropertyTable["RtcEtsProgOverwrite"]     = {PropertyId = 73, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["RtcMode"]                 = {PropertyId = 74, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RtcMode
g1.ExtendedPropertyTable["RtcTransmitCmdValue"]     = {PropertyId = 75, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["RtcCtrlModeHeating"]      = {PropertyId = 76, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RtcControllerMode
g1.ExtendedPropertyTable["RtcCtrlModeCooling"]      = {PropertyId = 77, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RtcControllerMode

g1.ExtendedPropertyTable["RtcHeatingType"]          = {PropertyId = 78, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RtcHeatingType
g1.ExtendedPropertyTable["RtcCoolingType"]          = {PropertyId = 79, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RtcCoolingType
g1.ExtendedPropertyTable["RtcHeatLowerLimit"]       = {PropertyId = 80, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //RtcLowerLimit
g1.ExtendedPropertyTable["RtcHeatUpperLimit"]       = {PropertyId = 81, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //RtcUpperLimit
g1.ExtendedPropertyTable["RtcCoolLowerLimit"]       = {PropertyId = 82, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //RtcLowerLimit
g1.ExtendedPropertyTable["RtcCoolUpperLimit"]       = {PropertyId = 83, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //RtcUpperLimit
g1.ExtendedPropertyTable["RtcHeatPropRange"]        = {PropertyId = 84, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //RtcPropRange
g1.ExtendedPropertyTable["RtcHeatResetTime"]        = {PropertyId = 85, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //RtcResetTime
g1.ExtendedPropertyTable["RtcCoolPropRange"]        = {PropertyId = 86, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //RtcPropRange
g1.ExtendedPropertyTable["RtcCoolResetTime"]        = {PropertyId = 87, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //RtcResetTime
g1.ExtendedPropertyTable["RtcOpModeViaSwitching"]   = {PropertyId = 88, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["RtcOpModeAfterReset"]     = {PropertyId = 89, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RtcHvacModeAfterReset
g1.ExtendedPropertyTable["RtcModeAfterReset"]       = {PropertyId = 90, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RtcModeAfterReset
g1.ExtendedPropertyTable["RtcProtectionDelay"]      = {PropertyId = 91, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //ProtectionDelay
g1.ExtendedPropertyTable["RtcSpHeatComfort"]        = {PropertyId = 92, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //SetpointTemp
g1.ExtendedPropertyTable["RtcSpHeatStandby"]        = {PropertyId = 93, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //SetpointTemp
g1.ExtendedPropertyTable["RtcSpHeatNight"]          = {PropertyId = 94, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //SetpointTemp
g1.ExtendedPropertyTable["RtcSpCoolComfort"]        = {PropertyId = 95, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //SetpointTemp
g1.ExtendedPropertyTable["RtcSpCoolStandby"]        = {PropertyId = 96, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //SetpointTemp
g1.ExtendedPropertyTable["RtcSpCoolNight"]          = {PropertyId = 97, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //SetpointTemp
g1.ExtendedPropertyTable["RtcSpAllow"]              = {PropertyId = 98, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["RtcSpMin"]                = {PropertyId = 123, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //SetpointTemp
g1.ExtendedPropertyTable["RtcSpMax"]                = {PropertyId = 124, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //SetpointTemp
g1.ExtendedPropertyTable["RtcSpStepWidth"]          = {PropertyId = 99, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //SetpointShift
g1.ExtendedPropertyTable["RtcSpHeatProtect"]        = {PropertyId = 100, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //SetpointTemp
g1.ExtendedPropertyTable["RtcSpFrostProtect"]       = {PropertyId = 101, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //SetpointTemp
g1.ExtendedPropertyTable["RtcSpTransDelay"]         = {PropertyId = 102, Type = 0x0A, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_FLOAT //SetpointTemp
g1.ExtendedPropertyTable["RtcSpTransCycle"]         = {PropertyId = 103, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //SetpointTransCycle
g1.ExtendedPropertyTable["RtcStatusTransDelta"]         = {PropertyId = 104, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //StatusTransDelta
g1.ExtendedPropertyTable["RtcStatusTransCycle"]         = {PropertyId = 105, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //SwitchTransCycle
g1.ExtendedPropertyTable["RtcStatusOutputTypeHeat"]     = {PropertyId = 106, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RtcOutputType
g1.ExtendedPropertyTable["RtcStatusOutputTypeCool"]     = {PropertyId = 107, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RtcOutputType
g1.ExtendedPropertyTable["RtcStatusCmdValueLimit"]      = {PropertyId = 108, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RtcCmdValueLimit
g1.ExtendedPropertyTable["RtcStatusCmdValueLimitReset"] = {PropertyId = 109, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RtcCmdValueLimitReset
g1.ExtendedPropertyTable["RtcStatusCmdValueMinHeat"]    = {PropertyId = 110, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //MinCommandValue
g1.ExtendedPropertyTable["RtcStatusCmdValueMaxHeat"]    = {PropertyId = 111, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //MaxCommandValue
g1.ExtendedPropertyTable["RtcStatusCmdValueMinCool"]    = {PropertyId = 112, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //MinCommandValue
g1.ExtendedPropertyTable["RtcStatusCmdValueMaxCool"]    = {PropertyId = 113, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //MaxCommandValue
g1.ExtendedPropertyTable["RtcStatusHeatingIndication"]  = {PropertyId = 114, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["RtcStatusCoolingIndication"]  = {PropertyId = 115, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["RtcStatusStatusType"]         = {PropertyId = 116, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RtcStatusType
g1.ExtendedPropertyTable["RtcStatusBehavior"]           = {PropertyId = 117, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RtcBehavior
g1.ExtendedPropertyTable["RtcCfPresence"]               = {PropertyId = 118, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RtcPresence
g1.ExtendedPropertyTable["RtcCfPresenceDuration"]       = {PropertyId = 119, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //RtcResetTime
g1.ExtendedPropertyTable["RtcCfLocked"]                 = {PropertyId = 120, Type = 0x35, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_ENUM8 //RtcIsLocked
g1.ExtendedPropertyTable["RtcCfSwitchOff"]              = {PropertyId = 121, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["RtcCfValveProtect"]           = {PropertyId = 122, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //Use
g1.ExtendedPropertyTable["RtcStatusAutoTrans"]          = {PropertyId = 125, Type = 0x02, ArrayCount = 1, InterfaceObjectId = InterfaceObjectIdRtc} -- PDT_UNSIGNED_CHAR //StatusTransCycle

m.InitExtendedPropertyTable(g1.ExtendedPropertyTable)

--[[
end KNX properties management
]]

return g1
