/*jslint browser: true, todo: true*/
/*global ko, Gira, $, _, DOMParser, WebSocket*/
(function (ns) {
    "use strict";
    var Log = Gira.Log,
        Globals = window.Gira.Globals;

    function ProxyConnectionWrapper(ip, port) {
        var self = this,
            createSocket = function () { return new WebSocket('ws://' + ip + ':' + port); },
            ws = createSocket(),
            emptyHandler = function () { return undefined; },
            handlers = {
                onopen: emptyHandler(),
                onclose: emptyHandler(),
                onerror: emptyHandler(),
                onmessage: emptyHandler()
            };
        self.send = function (message) {
            ws.send(message);
        };

        self.onopen = function (ea) {
            handlers.onopen = ea;
            ws.onopen = ea;
        };

        self.onclose = function (ea) {
            handlers.onclose = ea;
            ws.onclose = ea;
        };

        self.onerror = function (ea) {
            handlers.onerror = ea;
            ws.onerror = ea;
        };

        self.onmessage = function (ea) {
            handlers.onmessage = ea;
            ws.onmessage = ea;
        };

        self.readyState = function () {
            return ws.readyState;
        };

        self.close = function () {
            ws.close();
        };
        // Creates a new web socket and attaches the previously attached 'on' handlers
        self.renew = function () {
            ws = createSocket();
            ws.onopen = handlers.onopen;
            ws.onclose = handlers.onclose;
            ws.onerror = handlers.onerror;
            ws.onmessage = handlers.onmessage;
        };
    }

    function WeatherProxyConnectionListener(viewModelSubscribe) {
        var self = this,
            viewModel = viewModelSubscribe;

        self.onOpen = function () {
            viewModel._connectionOpen();
        };

        self.onClose = function () {
            return undefined;
        };

        self.onError = function () {
            return undefined;
        };

        self.onMessage = function (obj) {
            viewModel._apply(obj);
        };
    }

    function ProxyDataService(password, serviceClientID, serviceLanguage, connection) {
        var host = "http://homeserver.gira.de",
            weatherServicePassword = password,
            weatherServiceClientID = serviceClientID,
            commonParameters = "/dienste/wetter.xml?clientversion=2&lang={language}&jsonon=1",
            weatherCountriesUrl = "&query=laender&searchmethod=begins&searchstring={searchString}",
            weatherStationsUrl = "&query=wetterstationen&searchmethod=begins&searchstring={searchString}&land={landId}",
            weatherInfoUrl = "&query=wettervorhersage&ort_id={weatherStationId}",
            baseUrl,
            callbacks = {};

        baseUrl = host + commonParameters.replace("{language}", serviceLanguage);

        weatherInfoUrl = baseUrl + weatherInfoUrl;
        weatherCountriesUrl = baseUrl + weatherCountriesUrl;
        weatherStationsUrl = baseUrl + weatherStationsUrl;

        function getValue(obj) {
            if (obj instanceof Array) {
                if (obj.length > 0) {
                    return obj[0];
                }
                return undefined;
            }

            return obj;
        }

        function errorProcessing(ex, errorHandler, data) {
            Log.error("exception " + ex + " thrown for message " + JSON.stringify(data));
            if (errorHandler) {
                errorHandler(ex);
            }
        }

        function createCountryItem(data) {
            return {iso2lc: getValue(data.iso2lc), countryId: getValue(data.land_id), country: getValue(data.landname)};
        }
        function processWeatherCountriesRawObject(data) {
            var countries = data.land,
                result = [];

            if (countries) {
                if (countries instanceof Array) {
                    countries.forEach(function (item) {
                        result.push(createCountryItem(item));
                    });
                } else {
                    result.push(createCountryItem(countries));
                }
            }

            if (data.error) {
                Log.error("weather service returned error: " + JSON.stringify(data.error));
            }

            return result;
        }

        function createWeatherStationItem(item) {
            return {
                placeId: getValue(item.ort_id),
                place: getValue(item.ortsname),
                region: getValue(item.region),
                country: getValue(item.land),
                guid: Globals.createGUID()
            };
        }

        function processWeatherStationsRawObject(data) {
            var weatherStations = data.ort,
                result = [];

            if (weatherStations) {
                if (weatherStations instanceof Array) {
                    weatherStations.forEach(function (item) {
                        result.push(createWeatherStationItem(item));
                    });
                } else {
                    result.push(createWeatherStationItem(weatherStations));
                }
            }

            return result;
        }

        function processWeatherInfoRawObject(data) {
            var forecast = getValue(data.vorhersage),
                tags = forecast.tag,
                today = tags[0],
                resultToday,
                resultFutureDays = [];

            if (data.wetteronline) {
                return;
            }

            function getDay(tag) {
                var morning = tag.tagesabschnitte[0],
                    midday = tag.tagesabschnitte[1],
                    evening = tag.tagesabschnitte[2],
                    night = tag.tagesabschnitte[3];

                return {
                    date: getValue(tag.datum),
                    symbol: getValue(tag.symbol),
                    current: getValue(tag.tmax),
                    high: getValue(tag.tmax),
                    low: getValue(tag.tmin),
                    detail: {
                        rainfall: getValue(tag.niederschlagsmenge),
                        direction: getValue(tag.windrichtung),
                        humidity: getValue(tag.niederschlagswahrscheinlichkeit),
                        wind: getValue(tag.windgeschwindigkeit),

                        morning: {
                            symbol: getValue(morning.symbol),
                            high: getValue(morning.tmax),
                            low: getValue(morning.tmin)
                        },
                        midday: {
                            symbol: getValue(midday.symbol),
                            high: getValue(midday.tmax),
                            low: getValue(midday.tmin)
                        },
                        evening: {
                            symbol: getValue(evening.symbol),
                            high: getValue(evening.tmax),
                            low: getValue(evening.tmin)
                        },
                        night: {
                            symbol: getValue(night.symbol),
                            high: getValue(night.tmax),
                            low: getValue(night.tmin)
                        }
                    }
                };
            }

            resultToday = getDay(today);
            resultFutureDays.push(getDay(tags[1]));
            resultFutureDays.push(getDay(tags[2]));

            return {
                id: getValue(forecast.ort_id),
                town: getValue(forecast.ort),
                today : resultToday,
                forecasts : resultFutureDays
            };
        }

        function rawProcessing(data, rawProcesssor, successHandler, errorHandler) {
            try {
                var dataObj = rawProcesssor(data);
                if (successHandler) {
                    successHandler(dataObj);
                }
            } catch (ex) {
                errorProcessing(ex, errorHandler, data);
            }
        }

        function getInitCommandMessage(guid) {
            return {
                command: "Init",
                password: weatherServicePassword,
                clientId: weatherServiceClientID,
                guid: guid
            };
        }

        function getCommandMessage(command, url, guid) {
            return {
                command: command,
                url: url,
                guid: guid
            };
        }

        function sendDataRequest(url, guid, errorHandler) {
            var message = {};
            try {
                message = getCommandMessage("InvokeUrl", url, guid);
                connection.sendMessage(message);
            } catch (ex) {
                errorProcessing(ex, errorHandler, message);
            }
        }

        function ProxyDataServiceCall(processFunc, successHandler, errorHandler) {
            var guid = window.Gira.Globals.createGUID();

            function getGuid() {
                return guid;
            }

            return {
                getGuid: getGuid,
                process: processFunc,
                success: successHandler,
                error: errorHandler
            };
        }

        function getCountries(searchString, successHandler, errorHandler) {
            var url = weatherCountriesUrl.replace("{searchString}", encodeURIComponent(searchString)),
                proxyItem = new ProxyDataServiceCall(processWeatherCountriesRawObject, successHandler, errorHandler);
            callbacks[proxyItem.getGuid()] = proxyItem;
            sendDataRequest(url, proxyItem.getGuid(), errorHandler);
        }

        function getWeatherStations(countryId, searchString, successHandler, errorHandler) {
            var url = weatherStationsUrl.replace("{searchString}", encodeURIComponent(searchString))
                .replace("{landId}", countryId),
                proxyItem = new ProxyDataServiceCall(processWeatherStationsRawObject, successHandler, errorHandler);
            callbacks[proxyItem.getGuid()] = proxyItem;
            sendDataRequest(url, proxyItem.getGuid(), errorHandler);
        }

        function getWeather(weatherStationId, successHandler, errorHandler) {
            var url = weatherInfoUrl.replace("{weatherStationId}", weatherStationId),
                proxyItem;

            if (weatherStationId === undefined) {
                Log.error("Invalid weatherStationId!");
            }

            proxyItem = new ProxyDataServiceCall(processWeatherInfoRawObject, successHandler, errorHandler);
            callbacks[proxyItem.getGuid()] = proxyItem;
            sendDataRequest(url, proxyItem.getGuid(), errorHandler);
        }

        function connectionOpen() {
            connection.sendMessage(getInitCommandMessage(Gira.Globals.createGUID()));
        }

        function isValidData(data) {
            var error = data.response.error;
            if (error === undefined) {
                return true;
            }
            return error.state !== "ERROR" && error.error_msg === undefined;
        }

        function apply(data) {
            var client,
                guid;

            if (data.response === undefined || data.response.guid === undefined) {
                return;
            }

            guid = data.response.guid;
            if (guid.length > 0 && callbacks[guid] !== undefined) {
                client = callbacks[guid];

                try {
                    if (isValidData(data)) {
                        rawProcessing(data.response.value, client.process, client.success, client.error);
                    } else {
                        Log.exception(JSON.stringify(data));
                        if (client.error) {
                            client.error(undefined, data);
                        }
                    }
                    delete callbacks[guid];
                } catch (ex) {
                    if (callbacks[guid]) {
                        delete callbacks[guid];
                    }
                    errorProcessing(ex, client.error, data);
                }
            }
        }

        function getEmptyModel(weatherStationId, guid) {
            var emptyString = "--",
                defSymbol = "--",
                resultToday,
                resultFutureDays = [];

            function getDay() {
                return {
                    date: emptyString + "." + emptyString + "." + emptyString,
                    symbol: defSymbol,
                    current: emptyString,
                    high: emptyString,
                    low: emptyString,
                    detail: {
                        rainfall: emptyString,
                        direction: emptyString,
                        humidity: emptyString,
                        wind: emptyString,

                        morning: {
                            symbol: defSymbol,
                            high: emptyString,
                            low: emptyString
                        },
                        midday: {
                            symbol: defSymbol,
                            high: emptyString,
                            low: emptyString
                        },
                        evening: {
                            symbol: defSymbol,
                            high: emptyString,
                            low: emptyString
                        },
                        night: {
                            symbol: defSymbol,
                            high: emptyString,
                            low: emptyString
                        }
                    }
                };
            }

            resultToday = getDay();
            resultFutureDays.push(getDay());
            resultFutureDays.push(getDay());

            return {
                empty: true,
                guid: guid,
                id: weatherStationId,
                town: emptyString,
                today : resultToday,
                forecasts : resultFutureDays
            };
        }

        function init() {
            connectionOpen();
        }

        return {
            _connectionOpen: connectionOpen,
            _apply: apply,
            getCountries: getCountries,
            getWeatherStations: getWeatherStations,
            getWeather: getWeather,
            getEmptyModel: getEmptyModel,
            init: init
        };
    }

    ns.WeatherProxyConnectionListener = WeatherProxyConnectionListener;
    ns.ProxyConnectionWrapper = ProxyConnectionWrapper;
    ns.ProxyDataService = ProxyDataService;
}(Gira.Weather.Services));

//# sourceURL=weather.services.proxy.js
