/*jslint browser: true, todo: true*/
/*global ko, Gira, $, _, TouchEvent, TODO, moment*/
(function (ns) {
    "use strict";
    var Log = Gira.Log,
        Weather = window.Gira.Weather,
        touchStateFlags = {
            notInTouch: 0,
            inTouch: 1,
            inMove: 2
        };

    function WeatherConnectionListener(viewModel) {
        var self = this,
            SettingsKey = "weather.settings";

        self.onOpen = function () {
            return undefined;
        };

        self.onClose = function () {
            return undefined;
        };

        self.onError = function () {
            return undefined;
        };

        self.onMessage = function (obj) {
            var request;
            if (obj.event) {
                return;
            }
            if (obj.response && obj.response.request && obj.response.request.command === "SetAppValue") {
                request = obj.response.request;
                if (request.command === "SetAppValue" && request.key === SettingsKey) {
                    viewModel.recreateWeather();
                }
            }
        };
    }

    function WeatherSlideViewModel(data, weatherSymbols, defaultDateFormat) {
        var self = this,
            currentVisibleState = false,
            isEmptySlide = data.empty ? true : false,
            I18n = Weather.I18nTexts,
            touchState = {state: touchStateFlags.notInTouch, x: 0, y: 0},
            dateFormat = defaultDateFormat,
            rawDate;

        function checkIfIsEmptyData(empty) {
            isEmptySlide = empty ? true : false;
        }

        function getFixNumber(number) {
            var num;
            if (isEmptySlide) {
                return number;
            }
            num = parseFloat(number);
            return num.toFixed(0);
        }

        function getDayFromDate(date) {
            var dateObj;
            if (isEmptySlide) {
                return date;
            }
            dateObj = new Date(date);
            return I18n.daysText[dateObj.getDay()];
        }

        function getFormattedDate(date) {
            if (isEmptySlide) {
                return date;
            }
            return moment(date).format(dateFormat);
        }

        function onTouchStartHandler(state, event) {
            if (event.originalEvent instanceof TouchEvent) {
                state.x = event.originalEvent.touches[0].pageX;
                state.y = event.originalEvent.touches[0].pageY;
            } else {
                state.x = event.originalEvent.offsetX;
                state.y = event.originalEvent.offsetY;
            }
            state.state = touchStateFlags.inTouch;
        }

        function onTouchMoveHandler(state, event) {
            var x, y;
            if (state.state === touchStateFlags.inTouch) {
                if (event.originalEvent instanceof TouchEvent) {
                    x = event.originalEvent.touches[0].pageX;
                    y = event.originalEvent.touches[0].pageY;
                } else {
                    x = event.originalEvent.offsetX;
                    y = event.originalEvent.offsetY;
                }

                if (Math.abs(state.x - x) > 10) {
                    state.state = touchStateFlags.inMove;
                } else if (Math.abs(state.y - y) > 10) {
                    state.state = touchStateFlags.inMove;
                }
            }
        }

        function isInMove() {
            if (touchState.state === touchStateFlags.inMove) {
                touchState.state = touchStateFlags.notInTouch;
                touchStateFlags.x = touchStateFlags.y = 0;
                return true;
            }
            return false;
        }

        function ForecastItem(data, weatherSymbols, toogleCallback) {
            var forecastSelf = this,
                rawItemDate;

            forecastSelf.day = ko.observable();
            forecastSelf.date = ko.observable();
            forecastSelf.icon = ko.observable();
            forecastSelf.low = ko.observable();
            forecastSelf.high = ko.observable();
            forecastSelf.detail = {
                rainfall: ko.observable(),
                direction: ko.observable(),
                humidity: ko.observable(),
                wind: ko.observable()
            };

            forecastSelf.updateDateWithNewFormat = function () {
                forecastSelf.date(getFormattedDate(rawItemDate));
            };

            forecastSelf.isDetailVisible = ko.observable(false);
            forecastSelf.toggleTodayDetails = toogleCallback;

            forecastSelf.fillData = function (item) {
                forecastSelf.day(getDayFromDate(item.date));
                rawItemDate = item.date;
                forecastSelf.date(getFormattedDate(rawItemDate));
                forecastSelf.icon(weatherSymbols.getSmallIcon(item.symbol));
                forecastSelf.low(getFixNumber(item.low));
                forecastSelf.high(getFixNumber(item.high));

                forecastSelf.detail.rainfall(item.detail.rainfall);
                forecastSelf.detail.direction(item.detail.direction);
                forecastSelf.detail.humidity(item.detail.humidity);
                forecastSelf.detail.wind(item.detail.wind);
            };

            /*jslint unparam:true*/
            forecastSelf.onTouchStart = function (data, event) {
                onTouchStartHandler(touchState, event);
                return true;
            };

            forecastSelf.onTouchMove = function (data, event) {
                onTouchMoveHandler(touchState, event);
                return true;
            };
            /*jslint unparam:false*/

            forecastSelf.fillData(data);
        }

        self.guid = undefined;
        self.id = undefined;
        self.town = ko.observable();

        self.day = ko.observable();
        self.date = ko.observable();

        self.current = ko.observable();
        self.high = ko.observable();
        self.low = ko.observable();

        self.label1 = ko.observable();
        self.label2 = ko.observable();
        self.icon = ko.observable();

        self.detail = {
            rainfall: ko.observable(),
            direction: ko.observable(),
            humidity: ko.observable(),
            wind: ko.observable(),

            morningIcon: ko.observable(),
            morningLow: ko.observable(),
            morningHigh: ko.observable(),

            middayIcon: ko.observable(),
            middayLow: ko.observable(),
            middayHigh: ko.observable(),

            eveningIcon: ko.observable(),
            eveningLow: ko.observable(),
            eveningHigh: ko.observable(),

            nightIcon: ko.observable(),
            nightLow: ko.observable(),
            nightHigh: ko.observable()
        };

        self.forecasts = ko.observableArray();

        function hideAllForecasts() {
            self.forecasts().forEach(function (item) {
                item.isDetailVisible(false);
            });
        }

        function toggleTodayDetails(data) {
            if (isInMove()) {
                return;
            }

            self.isDetailVisible(false);
            currentVisibleState = data.isDetailVisible();
            hideAllForecasts();
            data.isDetailVisible(!currentVisibleState);
        }

        function updateDateWithNewFormat() {
            self.date(getFormattedDate(rawDate));
            self.forecasts().forEach(function (item) {
                item.updateDateWithNewFormat(dateFormat);
            });
        }

        self.hideAllDetails = function () {
            self.isDetailVisible(false);
            self.forecasts().forEach(function (item) {
                item.isDetailVisible(false);
            });
        };

        self.fillData = function (data) {
            var text,
                today = data.today,
                detail = today.detail,
                forecasts = data.forecasts;

            checkIfIsEmptyData(data.empty);

            self.guid = data.guid;
            self.id = data.id;
            self.town(data.town);

            self.day(I18n.mainDisplayTodayText);
            rawDate = today.date;
            self.date(getFormattedDate(rawDate));

            self.current(getFixNumber(today.current));
            self.high(getFixNumber(today.high));
            self.low(getFixNumber(today.low));

            text = weatherSymbols.getText(today.symbol);
            self.label1(text.text1);
            self.label2(text.text2);
            self.icon(weatherSymbols.getBigIcon(today.symbol));

            // function 0,1 -> 10
            self.detail.rainfall(detail.rainfall);
            self.detail.direction(detail.direction);
            self.detail.humidity(detail.humidity);
            self.detail.wind(detail.wind);

            self.detail.morningIcon(weatherSymbols.getBlueIcon(detail.morning.symbol));
            self.detail.morningLow(getFixNumber(detail.morning.low));
            self.detail.morningHigh(getFixNumber(detail.morning.high));

            self.detail.middayIcon(weatherSymbols.getBlueIcon(detail.midday.symbol));
            self.detail.middayLow(getFixNumber(detail.midday.low));
            self.detail.middayHigh(getFixNumber(detail.midday.high));

            self.detail.eveningIcon(weatherSymbols.getBlueIcon(detail.evening.symbol));
            self.detail.eveningLow(getFixNumber(detail.evening.low));
            self.detail.eveningHigh(getFixNumber(detail.evening.high));

            self.detail.nightIcon(weatherSymbols.getBlueIcon(detail.night.symbol));
            self.detail.nightLow(getFixNumber(detail.night.low));
            self.detail.nightHigh(getFixNumber(detail.night.high));

            if (self.forecasts().length === 0) {
                forecasts.forEach(function (item) {
                    self.forecasts.push(new ForecastItem(item, weatherSymbols, toggleTodayDetails));
                });
            } else {
                self.forecasts()[0].fillData(forecasts[0]);
                self.forecasts()[1].fillData(forecasts[1]);
            }
        };

        self.isDetailVisible = ko.observable(false);

        self.toggleTodayDetails = function () {
            if (isInMove()) {
                return;
            }

            currentVisibleState = self.isDetailVisible();
            hideAllForecasts();
            self.isDetailVisible(!currentVisibleState);
        };

        /*jslint unparam:true*/
        self.onTouchStart = function (data, event) {
            onTouchStartHandler(touchState, event);
            return true;
        };

        self.onTouchMove = function (data, event) {
            onTouchMoveHandler(touchState, event);
            return true;
        };
        /*jslint unparam:false*/

        self.setDataFormat = function (value) {
            dateFormat = value;
            updateDateWithNewFormat();
        };

        self.fillData(data);
    }

    ns.WeatherSlideViewModel = WeatherSlideViewModel;

    function MainDisplayViewModel(weatherSymbolsObj, theBrandingTitle, theConfig, systemMessageOpenWarning, dateTimeFormats) {
        var self = this,
            dataService,
            weatherSymbols = weatherSymbolsObj,
            settings = null,
            config = theConfig,
            openWarning = systemMessageOpenWarning,
            dateFormat = "DD.MM.YYYY";

        self.brandingTitle = theBrandingTitle;
        self.towns = ko.observableArray(
            ko.utils.arrayMap(undefined, function (item) {
                return ko.observable(item);
            })
        );
        self.messageText = ko.observable(false);
        self.showMessage = ko.observable(false);

        self.setDataService = function (dataServiceRest) {
            dataService = dataServiceRest;
        };

        function hideAllWeatherStation() {
            self.towns().forEach(function (item) {
                item.hideAllDetails();
            });
        }

        function clearCollection() {
            self.towns.removeAll();
        }

        function updateTown(town, data) {
            town.fillData(data);
        }

        function createTown(data) {
            self.towns.push(new ns.WeatherSlideViewModel(data, weatherSymbols, dateFormat));
        }

        function processTownData(data) {
            var town = _.filter(self.towns(), function (item) {
                return item.id === data.id && item.guid === data.guid;
            });

            if (town.length > 0) {
                town.forEach(function (item) {
                    updateTown(item, data);
                });
            } else {
                createTown(data);
            }
        }

        function addDummyWeatherInfo(emptyModel) {
            processTownData(emptyModel);
        }

        self.recreateWeather = function () {
            if (!dataService) {
                return;
            }

            settings = Weather.weatherSettings.getCurrentSettings();

            if (settings) {
                clearCollection();

                if (settings.weatherStations.length > 0) {
                    addDummyWeatherInfo(dataService.getEmptyModel(settings.weatherStations[0].weatherStationId, settings.weatherStations[0].guid));
                    dataService.getWeather(settings.weatherStations[0].weatherStationId, function (data) {
                        data.guid = settings.weatherStations[0].guid;
                        processTownData(data);
                        self.onAfterTownsChanged();
                        self.showMessage(false);
                    }, self.errorHandler);
                } else {
                    self.showEmptyResultMessage();
                }

                setTimeout(function () {
                    settings.weatherStations.forEach(function (item, index) {
                        if (index > 0) {
                            dataService.getWeather(item.weatherStationId, function (data) {
                                data.guid = item.guid;
                                processTownData(data);
                                self.onAfterTownsChanged();
                            });
                        }
                    }, self.errorHandler);
                }, 1000);
            }
            self.onAfterTownsChanged();
        };

        self.showEmptyResultMessage = function () {
            self.messageText(config.text.weatherEmptyConfigMessageText);
            self.showMessage(true);
        };

        self.errorHandler = function (ex, data) {
            try {
                if (data || ex) {
                    openWarning(config.text.weatherServiceUnavailableMessageText);
                }
            } catch (e) {
                Log.exception(e);
            }
        };

        self.init = function () {
            dateFormat = dateTimeFormats.chosenDateFormat();
            self.recreateWeather();
            dateTimeFormats.chosenDateFormat.subscribe(function (value) {
                self.towns().forEach(function (item) {
                    dateFormat = value;
                    item.setDataFormat(dateFormat);
                });
            });
        };

        self.afterMove = function () {
            hideAllWeatherStation();
        };

        self.onAfterTownsChanged = function () {
            if (self.afterTownsChanged !== undefined) {
                self.afterTownsChanged();
            }
        };

        self.afterTownsChanged = undefined;
    }

    ns.WeatherConnectionListener = WeatherConnectionListener;
    ns.MainDisplayViewModel = MainDisplayViewModel;
}(Gira.Weather.ViewModels));

//# sourceURL=weather.display.viewmodel.js
