/*global window: true, Gira, $, _*/
/*jslint browser: true*/

window.Gira = window.Gira || {};
window.Gira.Demo = window.Gira.Demo || {};

(function (ns) {
    "use strict";
    var Log = Gira.Log;

    function DemoWeatherProxy(sendResponseCallback) {
        var staticData,
            settings = {acceptedLicense: true, weatherStations: []},
            sendWithoutLogging = function (message) {
                sendResponseCallback(message);
            },
            sendResponse = function (messageId, command, msg) {
                var stringMessage = '{"response": {"guid": "' + messageId + '","command": "' + command + '","value": ' + msg + ',"error": {"state": "OK","text": ""}}}';
                sendWithoutLogging(stringMessage);
            },
            sendGdsResponse = function (request, msg) {
                var stringMessage = '{"response": {"request": ' + JSON.stringify(request) + ',"value": ' + msg + ',"error": {"state": "OK","text": ""}}}';
                sendWithoutLogging(stringMessage);
            };

        function initSettingsFromStaticData() {
            var countryNameToId = {};
            _.each(staticData.countries.land, function (country) {
                countryNameToId[country.landname] = country.land_id;
            });
            settings.weatherStations = _.map(staticData.weatherStations.ort, function (station) {
                return {
                    countryId: countryNameToId[station.land],
                    weatherStationId: station.ort_id,
                    label: station.ortsname,
                    guid: window.Gira.Globals.createGUID()
                };
            });
        }

        function loadStaticData(loadedCallback) {
            if (staticData) {
                loadedCallback(staticData);
            } else {
                var data = $.ajax({
                        url: "weather/demo/data/demoData.json",
                        async: false
                    }).responseText;
                staticData = JSON.parse(data);
                initSettingsFromStaticData();
                loadedCallback(staticData);
            }
        }

        function searchCountries(data, searchString) {
            var result = _.select(data.countries.land, function (item) {
                return item.landname.toLowerCase().indexOf(searchString.toLowerCase()) >= 0;
            });
            return result === undefined ? {} : {land: result};
        }

        function searchWeatherStations(data, parameters, searchString) {
            var landParameter,
                landId,
                land,
                result;
            landParameter = _.find(parameters, function (item) {
                return item.indexOf("land") >= 0;
            });
            if (landParameter) {
                landId = landParameter.split("=")[1];
                land =  _.find(data.countries.land, function (item) {
                    return item.land_id.indexOf(landId) >= 0;
                });
            }
            if (land) {
                result  = _.select(data.weatherStations.ort, function (item) {
                    return (item.ortsname.toLowerCase().indexOf(searchString.toLowerCase()) >= 0) &&
                           (item.land.toLowerCase().indexOf(land.landname.toLowerCase()) >= 0);
                });
            }
            return result === undefined ? {} : {ort: result};
        }

        function searchWeatherInfo(data, searchString) {
            var result = _.find(data.weatherInfo, function (item) {
                return item.vorhersage.ort_id === searchString;
            });
            return result === undefined ? {} : result;
        }

        function getSearch(parameters, queryType) {
            var search;
            switch (queryType) {
            case "laender":
            case "wetterstationen":
                search = _.find(parameters, function (item) {
                    return item.indexOf("searchstring") >= 0;
                });
                break;
            case "wettervorhersage":
                search = _.find(parameters, function (item) {
                    return item.indexOf("ort_id") >= 0;
                });
                break;
            }
            return search.split("=");
        }

        function performRequest(path, success) {
            var parameters = path.split("&"),
                queryType,
                searchString,
                query;
            loadStaticData(function (data) {
                if (data) {
                    query = _.find(parameters, function (item) {
                        return item.indexOf("query") >= 0;
                    });

                    if (query) {
                        queryType = query.split("=");
                        searchString = getSearch(parameters, queryType[1]);
                        switch (queryType[1]) {
                        case "laender":
                            success(searchCountries(data, searchString[1]), 'countries');
                            break;
                        case "wetterstationen":
                            success(searchWeatherStations(data, parameters, searchString[1]), 'weatherStations');
                            break;
                        case "wettervorhersage":
                            success(searchWeatherInfo(data, searchString[1]), 'weatherInfo');
                            break;
                        }
                    }
                } else {
                    Log.error("Static request without loaded static data!");
                }
            });
        }

        function processGdsMessages(msg) {
            if (msg.request.command === "GetAppValue") {
                sendGdsResponse(msg.request, JSON.stringify(JSON.stringify(settings))); // has to be a string within a string
            } else if (msg.request.command === "SetAppValue" && msg.request.key === "weather.settings") {
                settings = JSON.parse(msg.request.value);
                sendGdsResponse(msg.request, JSON.stringify(settings));
            }
        }

        function processMessage(msg) {
            if (msg.command === "InvokeUrl") {
                performRequest(msg.url, function (data) {
                    sendResponse(msg.guid, msg.command, JSON.stringify(data));
                });
            } else {
                processGdsMessages(msg);
            }
        }

        function receive(message) {
            processMessage(message);
        }

        // eagerly load the static data so that the settings can be initialized
        loadStaticData(function () { return undefined; });

        return {
            receive: receive
        };
    }

    ns.DemoWeatherProxy = DemoWeatherProxy;

}(Gira.Demo));

//# sourceURL=demo.weather.proxy.js