/*global $, tsx*/
/*jslint browser: true*/

/**
 * The Class tsx.Popup handles the Popup Window
 * 
 */
tsx.Popup = (function () {

/** 
* @enum ModeTypes
*/
    var ModeType = {
        SUPER: 0,
        ERROR: 1,
        WARNING: 2,
        INFO: 3,
        OKAY: 4
    };

    var popup = {},
        I18nTexts = window.Gira.appLang.appI18nTexts;
    popup.ModeType = ModeType;
    var mLoad = false;
    var mCurrentMode = undefined;
    var mButtonOne;
    var mButtonTwo;
    var $btnLeft, $btnRight;
    var mHidden = true;
    var radio;
    var icon;


    var messageArray = [];
    messageArray[ModeType.ERROR] = [];
    messageArray[ModeType.SUPER] = [];
    messageArray[ModeType.INFO] = [];
    messageArray[ModeType.OKAY] = [];
    messageArray[ModeType.WARNING] = [];

    function createMessage(channel, text) {
        return {channel: channel, text: text};
    }

    function findMessageIndexByChannel(messages, channel) {
        for (var i = 0; i < messages.length; i++) {
            if (messages[i].channel == channel) {
                return i;
            }
        }
        return -1;
    }

    /**
     * Checks if messages are empty.
     */
    popup.isEmpty = function () {
        return messageArray[ModeType.ERROR].length === 0 //
            && messageArray[ModeType.SUPER].length === 0 //
            && messageArray[ModeType.INFO].length === 0 //
            && messageArray[ModeType.OKAY].length === 0 //
            && messageArray[ModeType.WARNING].length === 0; //
    };

    /**
     * Initialize the Popup Window
     * with Mode, who is the Owner of the Popup and 
     * which Callback Functions have the Buttons when it is Mode "info"
     *
     * @param {int} mode 
     *          The Popup Mode (0 = error, 1 = warning, 2 = info, 3 = okay)
     * @param {function} buttonOne 
     *          Callback Function for buttonOne (right button)
     * @param {function} buttonTwo 
     *          Callback Function for buttonTwo (left button)
     */
    popup.init = function (mode, buttonOne, buttonTwo) {
        // loading the popup from file
        mButtonOne = buttonOne;
        mButtonTwo = buttonTwo;

        if (!mLoad) {
            load();
            //popupxsl function
            $('#popup').data('settingsSwiper').setWrapperTranslate(0, 0, 0);
            $btnLeft = $("#popup > .popup-foreground > .buttons > .left");
            $btnRight = $("#popup > .popup-foreground > .buttons > .right");
            mLoad = true;
        }       
    };

    /**
     * addMessage to the Popup on Top of the Message Array
     * and the Popuplist
     *
     * @param {int} id 
     *          Message ID
     * @param {int} mode 
     *          The Popup Mode (0 = error, 1 = warning, 2 = info, 3 = okay)
     * @param {string} message
     *          Message to display
     */
    popup.addMessage = function (id, mode, message) {
        switch (mode) {
        case ModeType.ERROR:
            var index = findMessageIndexByChannel(messageArray[mode], id);
            if (index >= 0) {
                messageArray[mode][index].text = message;
            } else {
                messageArray[mode].unshift(createMessage(id, message));
            }
            break;
        case ModeType.SUPER:
        case ModeType.INFO:
        case ModeType.OKAY:
        case ModeType.WARNING:
            messageArray[mode][0] = createMessage(0, message);
            break;
        default:
            break;
        }

        switch (mode) {
        case ModeType.SUPER:
        case ModeType.ERROR:
            if (mCurrentMode!=undefined && (mCurrentMode != ModeType.ERROR)) {
                break;
            }
        case ModeType.INFO:
        case ModeType.OKAY:
        case ModeType.WARNING:
        default :
                // this is only when an error popup is open and a user popup is coming up at the "same" time
            if (!mHidden) {
                if (mode != mCurrentMode && !(mode == ModeType.SUPER && mCurrentMode == ModeType.ERROR)) {
                    tsx.Popup.closePopup();
                    $("#popup > .popup-foreground").on('webkitTransitionEnd otransitionend oTransitionEnd msTransitionEnd transitionend', function() {
                        build(mode);
                        popup.show();
                    });
                } else {
                    build(mode);
                }
            } else {
                build(mode);
                popup.show();
            }
            break;
        }
    };

    /**
     * removeMessage from the Popup and the Message Array
     *
     * @param {int} id 
     *          Message ID
     */
    popup.removeMessage = function (id, mode, animationEndCallback) {
        switch (mode) {
            case ModeType.ERROR:
                var index = findMessageIndexByChannel(messageArray[mode], id);
                if (index >= 0) {
                    messageArray[mode].splice(index, 1);
                    if (mCurrentMode == ModeType.ERROR) {
                        if (messageArray[ModeType.ERROR].length > 0) {
                            setMessage(ModeType.ERROR);
                        } else {
                            //close the popup and unbind events on statusbar if there is no message left
                            if (messageArray[ModeType.SUPER].length == 0) {
                                unbindStatusbar();
                                tsx.Popup.closePopup(animationEndCallback);
                            }
                        }
                    }
                }
                break;
            case ModeType.SUPER:
                if (messageArray[mode].length > 0) {
                    messageArray[mode] = [];

                    if (mCurrentMode == ModeType.SUPER) {
                        //close popup if no ERROR (not super error) is left
                        if (messageArray[ModeType.ERROR].length === 0) {
                            //close the popup and unbind events on statusbar if there is no message left
                            unbindStatusbar();
                            tsx.Popup.closePopup(animationEndCallback);
                        } else {
                            setMessage(ModeType.ERROR);
                        }
                    } else {
                        if (messageArray[ModeType.ERROR].length === 0) {
                            //close the popup and unbind events on statusbar if there is no message left
                            unbindStatusbar();
                        }
                    }
                }
                break;
            case ModeType.INFO:
            case ModeType.OKAY:
            case ModeType.WARNING:
                messageArray[mode] = [];
                break;
            default:
                break;
            }
    };

    /**
     * Animates the Popup and make it Visible
     */
    popup.show = function () {
        NavbarController.getInstance()._addUnclickableOverlay();
        if (mHidden) {
            $("#popup").css("visibility", "visible");
            $("#popup > .popup-background").removeClass("animation-reverse");
            $("#popup > .popup-foreground").removeClass("slideOutFromBottom-popupAnimation");
        }
        $("#popup > .popup-background").addClass("animation");
        $("#popup > .popup-foreground").addClass("slideInFromBottom-popupAnimation");
        $("#popup > .popup-foreground").on('webkitTransitionEnd otransitionend oTransitionEnd msTransitionEnd transitionend', function () {
            NavbarController.getInstance()._removeUnclickableOverlay();
            unbindStatusbar();
            mHidden = false;

        });
    };

    /**
     * close the popup
     *
     * @param {function} animationEndCallback 
     *          Function which is called after the popup close animation is completed
     */
    popup.closePopup = function (animationEndCallback) {
        //close
        //animates the popup down and the black background to transparent
        $("#popup > .popup-background").removeClass("animation").addClass("animation-reverse");
        $("#popup > .popup-foreground").removeClass("slideInFromBottom-popupAnimation").addClass("slideOutFromBottom-popupAnimation");

        if (!mHidden) {
            //wait for transition end
            $("#popup > .popup-foreground").on('webkitTransitionEnd otransitionend oTransitionEnd msTransitionEnd transitionend', function() {
                //hides the popup
                $("#popup").css("visibility", "hidden");
                bindStatusbar();
                mHidden = true;
                //destroy the popup in the dom tree
                destroy();
                if (typeof animationEndCallback == "function") {
                        animationEndCallback();
                }
                $("#popup > .popup-foreground").off();
            });
        } else if (mHidden) {
            destroy();
        }
    };

    /**
     * destroys the popup in the dom tree if there is no message left
     */
    function destroy() {
        if (messageArray[ModeType.SUPER].length > 0) {
            build(ModeType.SUPER);
        } else if (messageArray[ModeType.ERROR].length > 0) {
            build(ModeType.ERROR);
        } else {
            unbindStatusbar();
            $("#popup").remove();
            mLoad = false;
            mCurrentMode = undefined;
        }
    }

    /**
     * build the popup
     *
     * @param {int} mode 
     *          The Popup Mode (0 = error, 1 = warning, 2 = info, 3 = okay)
     */
    function build(mode) {
        setMessage(mode);
        //chose the mode
        switch (mode) {
        case ModeType.SUPER:
            createError();
            break;
        case ModeType.ERROR:
            createError();
            break;
        case ModeType.WARNING:
            createWarning();
            break;
        case ModeType.INFO:
            createInfo();
            break;
        case ModeType.OKAY:
            createOkay();
            break;
        }
        setIcon();
    }

    /**
      * build the error mode
      */
    function createError() {
        radio = "button-red";
        icon = popup.mWarningIcon;
        setButton($btnRight, I18nTexts.menuConfigOk, function () { fsm.API.SystemMessage.Minimize(); });
    }

    /**
     * build the warning mode
     */
    function createWarning() {
        radio = "button-red";
        icon = popup.mWarningIcon;
        setButton($btnRight, I18nTexts.menuConfigOk, function () { fsm.API.SystemMessage.CloseWarning(); });
    }

    /**
    * build the info mode
    */
    function createInfo() {
        radio = "button-grey";
        icon = popup.mInfoIcon;
        if (typeof mButtonOne == "function") {
            setButton($btnRight, I18nTexts.menuConfigOk, mButtonOne);
        }
        if (typeof mButtonTwo == "function") {
            setButton($btnLeft, I18nTexts.menuConfigCancel , mButtonTwo);
        }
    }

    /**
    * build the okay mode
    */
    function createOkay() {
        radio = "button-green";
        icon = popup.mOkayIcon;
        setButton($btnRight, I18nTexts.menuConfigOk, function () { fsm.API.SystemMessage.CloseOkay(); });
    }

    /**
     * initiliaze buttons
     *
     * @param {jquery element} btn 
     *          The Button
     * @param {string} message 
     *          The text on the Button
     * @param {function} fn 
     *          Callback Function for buttonOne (right button)
     */
    function setButton($btn, message, fn) {
        $btn.text(message);
        $btn.html(function () {
            registerEventsForPressButton($btn,
            function () {
                $btn.addClass('down');
            }, function () {
                $btn.removeClass('down');
            }, 
               fn
            );
        });
        $btn.show();
    }

    /**
     * writes the text messages on the top of the popup
     */
    function setMessage(mode) {
        mCurrentMode = mode;
        var messageDiv = "";
        var i = 1;
        for (var message in messageArray[mode]) {
            messageDiv += "<div class=\"message\">";
            messageDiv += "<div class=\"text t2sw\">";
            messageDiv += messageArray[mode][message].text;
            messageDiv += "</div>";
            messageDiv += "<div class=\"icon\">";
            messageDiv += "<div class=\"radio\"></div>";
            messageDiv += "<div class=\"sprites\"></div>";
            messageDiv += "</div>";
            if (i < messageArray[mode].length) {
                messageDiv += "<div class=\"popup-double-line\"></div>";
            }
            messageDiv += "</div>";
            i++;
        }
        var messageBox = $("#popup > .popup-foreground .message-box");
        messageBox.html(messageDiv);
        setIcon();
    }

    /*
     * displays the icon in the popup
     */
    function setIcon() {
        var messageBox = $("#popup > .popup-foreground .message-box");
        messageBox.find(".radio").attr('class', 'radio ' + radio);
        messageBox.find(".sprites").attr('class', 'sprites ' + icon);
    }

    /*
     * bind Statusbar
     */
    function bindStatusbar() {
        var statusbar = $("#statusbar");
        statusbar.on('touchend click', function () {
            fsm.API.SystemMessage.Maximize();
        });
    }

    /*
     * bind Statusbar
     */
    function unbindStatusbar() {
        var statusbar = $("#statusbar");
        statusbar.off();
    }

    /*
     * ajax call to load the popup template
     */
    function load() {
        $.ajax({
            url: "templates/popup.xhtml",
            data: {},
            cache: false,
            async: false,
            success: function (data) {
                $("body").append($(data));
            },
            dataType: 'html'
        });
    }

    return popup;
}());