/*global $, ComClient, registerEventsForSwipeableElement*/
/*jslint browser: true*/

/**
 * New DimmerAbsoluteController element with different parameters
 * Init DimmerAbsoluteController function and bind events to knob and buttons
 * @param {String} knobId id of the knob element
 * @param {String} tagId tag from config
 * @param {String} tagStatusId tag from config
 * @param {String} onofftagId tag from config
 * @param {String} decId id of decrease button
 * @param {String} incId id of increase button
 * @param {Integer} step step value from config
 * @param {String} containerId if of whole container element
 * @param {String} hatch image for the dimmer hatch
 * @param {String} ring image for the dimmer ring
 * @param {Integer} animationSpeed duration to animate a value change of one percent
 * @param {Integer} inUseTimer duration during whih user tag values are ignored after user input
 */
function DimmerAbsoluteController(knobId, tagId, tagStatusId, onofftagId, decId, incId, buttonId, step, containerId, buttonConfigPlus, buttonConfigMinus, buttonConfig, hatch, ring, animationSpeed, inUseTimer) {
    var button = this;
    var tag = this;
    var tagStatus = this;
    var onofftag = this;
    this._inUse = false;
    this._inUseTimeout = undefined;
    this._inUseTimer = inUseTimer;

    this._knobId = knobId;
    this._decId = decId;
    this._incId = incId;
    this._buttonId = buttonId;
    this._steps = step;
    this._containerId = containerId;
    this._tagId = tagId;
    this._tagStatusId = tagStatusId;
    this._onofftagId = onofftagId;
    this._buttonConfigPlus = buttonConfigPlus;
    this._buttonConfigMinus = buttonConfigMinus;
    this._buttonConfig = buttonConfig;

    this._container = $(this._containerId);
    this._dec = this._container.find(this._decId);
    this._inc = this._container.find(this._incId);
    button = this._container.find(this._buttonId);
    this._knobs = this._container.find(this._knobId);
    tag = this._tagId;
    tagStatus = this._tagStatusId;
    onofftag = this._onofftagId;

    this._min = 0;
    this._max = 100;

    longPressInterval = null;
    longPressTimeout = null;

    var $knobInput = this._container.find('input');
    var $bg = this._container.find('.bg');
    $knobInput.attr("data-width", parseInt($bg.css("width"), 10));
    $knobInput.attr("data-height", parseInt($bg.css("height"), 10));

    var actualImage = new Image();
    var setpointImage = new Image();
    actualImage.src = 'img/' + ring.slice(ring.indexOf('_') + 1) + '.png';
    setpointImage.src = 'img/' + hatch.slice(ring.indexOf('_') + 1) + '.png';

    var _this = this;

    /**
     * Defines functions on knob change and release
     */
    this.knobInstance = $(this._knobId).knob({
        change: function(val) {
            // if choosen value is bigger than max value minus steps, change to max value
            if (val > _this._max) {
                val = _this._max;
            }
            // if choosen value is smaller than min value plus steps, change to min value
            if (val < _this._min) {
                val = _this._min;
            }
            // set choosen value
            _this.setPercent(val);
        },
        setpoint: setpointImage,
        actual: actualImage,
        animationSpeed: animationSpeed,
        sendValue: function(value) {
            ComClient.sendTagValue(tag, value);
            _this.setInputValue(value);
        }

    });

    // Bind events on buttons and knob
    this.bindEvents(this._container, this._inc, this._dec, button, tag, tagStatus, onofftag, this._knobs);
}

/**
 * Updates button css class on percent value
 * If percent smaller or equal min value, decrease button will be inactive
 * If percent greater or equal max value, increase button will be inactive * 
 * @param {Integer} percent value of knob
 */
DimmerAbsoluteController.prototype.updateIncDecButtons = function (percent) {

    var decs = this._dec.find('.decs');
    var incs = this._inc.find('.incs');

    // value smaller or equal min
    if (this._min >= percent) {
        decs.removeClass(this._buttonConfigMinus);
        decs.addClass(this._buttonConfigMinus + "_inactive");
        decs.removeClass(this._buttonConfigMinus + "_pushed");
        this._dec.removeClass("down");
        incs.removeClass(this._buttonConfigPlus + "_inactive");
        incs.addClass(this._buttonConfigPlus);
        this._dec.attr('disabled', true);
        this._inc.removeAttr('disabled');
        this.releaseLongPress();
    }

    // value greater or equal max
    if (this._max <= percent) {
        incs.removeClass(this._buttonConfigPlus);
        incs.addClass(this._buttonConfigPlus + "_inactive");
        incs.removeClass(this._buttonConfigPlus + "_pushed");
        decs.removeClass(this._buttonConfigMinus + "_inactive");
        this._inc.removeClass("down");
        decs.addClass(this._buttonConfigMinus);
        this._inc.attr('disabled', true);
        this._dec.removeAttr('disabled');
        this.releaseLongPress();
    }

    // value between min and max
    if (this._max > percent && this._min < percent) {
        var ele = incs[0];
        var cName = ele.className;
        var sprite = cName.match(/\bsprite_\w*_inactive/i);
        if (sprite !== null) {
            var spriteNew = sprite[0].replace("_inactive", "");
            ele.className = cName.replace(sprite[0], spriteNew);
        }
        ele = decs[0];
        cName = ele.className;
        sprite = cName.match(/\bsprite_\w*_inactive/i);
        if (sprite !== null) {
            spriteNew = sprite[0].replace("_inactive", "");
            ele.className = cName.replace(sprite[0], spriteNew);
        }
        this._inc.removeAttr('disabled');
        this._dec.removeAttr('disabled');
    }

};

/**
 * Updates on off text on percent value
 * If min value smaller percent, on text will be active
 * If min value greater or equal percent, off text will be shown 
 * @param {Integer} percent value of knob
 */
DimmerAbsoluteController.prototype.updateOnOffText = function (percent) {
    var on = this._container.find('.dimmer-text-on');
    var off = this._container.find('.dimmer-text-off');

    on.clearQueue();
    off.clearQueue();

    if (this._min < percent) {
        on.animate({ opacity: 1 }, 500);
        off.animate({ opacity: 0 }, 500);
    }
    if (this._min >= percent) {
        on.animate({ opacity: 0 }, 500);
        off.animate({ opacity: 1 }, 500);
    }
};

/**
 * Increase percent value and trigger change on knob
 */
DimmerAbsoluteController.prototype.incPercent = function () {
    var newValue = Round(this._knobs.val(), 0) + parseInt(this._steps);
    if (newValue > this._max) newValue = this._max;
    ComClient.sendTagValue(this._tagId, newValue);
    this.setInputValue(newValue);
};

/**
 * Decrease percent value and trigger change on knob
 */
DimmerAbsoluteController.prototype.decPercent = function() {
    var newValue = Round(this._knobs.val(), 0) - parseInt(this._steps);
    if (newValue < this._min) newValue = this._min;
    ComClient.sendTagValue(this._tagId, newValue);
    this.setInputValue(newValue);
};

/**
 * Set percent value and update buttons
 * @param {Integer} percent value of knob
 */
DimmerAbsoluteController.prototype.setPercent = function (percent) {
    this._container.find('.set-percent').text(percent);
    this.updateIncDecButtons(percent);
    this.updateOnOffText(percent);
};

/**
 * Defines long press start behavior
 * @param {function} fnc function to call at long press
 */
DimmerAbsoluteController.prototype.startLongPress = function (fnc) {
    clearInterval(longPressInterval);
    clearTimeout(longPressTimeout);
    longPressTimeout = setTimeout(function () {
        longPressInterval = setInterval(function () {
            fnc();
        }, 200);
    }, 300);
};

/**
 * Defines long press end behavior
 */
DimmerAbsoluteController.prototype.releaseLongPress = function () {
    setTimeout(function () {
        clearInterval(longPressInterval);
        clearTimeout(longPressTimeout);
        longPressInterval = null;
    }, 5);
};

/**
 * Bind events on buttons and canvans
 * @param {Element} container container element 
 * @param {Element} inc increase button element
 * @param {Element} dec decrease button element
 * @param {String} tag tag from config
 * @param {String} tagStatus tag from config
 * @param {String} onofftag tag from config
 * @param {Element} knobs knob element
 */
DimmerAbsoluteController.prototype.bindEvents = function (container, inc, dec, button, tag, tagStatus, onofftag, knobs) {
    var _this = this;

    // Events on increase button
    registerEventsForSwipeableElementKnob(container.find('.inc'),
       function () {
           if (inc.attr('disabled') === undefined) {
               var ele = inc.find('.incs')[0];
               var cName = ele.className;
               var sprite = cName.match(/\bsprite_\w*/i);
               if (sprite !== null && sprite[0].match("_pushed") === null) {
                   var spriteNew = sprite[0] + "_pushed";
                   ele.className = cName.replace(sprite[0], spriteNew);
                   inc.addClass('down');
               }
               _this.startLongPress(function () {
                   _this.incPercent(container, inc, dec, tag);
               });
           }
       },
       function () {
           if (inc.attr('disabled') === undefined) {
               var ele = inc.find('.incs')[0];
               var cName = ele.className;
               var sprite = cName.match(/\bsprite_\w*_pushed/i);
               if (sprite !== null) {
                   var spriteNew = sprite[0].replace("_pushed", "");
                   ele.className = cName.replace(sprite[0], spriteNew);
                   inc.removeClass('down');
               }
               _this.releaseLongPress();
           }
       },
       function () {
           if (inc.attr('disabled') === undefined) {
               if (!longPressInterval) {
                   _this.incPercent(container, inc, dec, tag);
               }
           }
       }
   );


    // Events on decrease button
    registerEventsForSwipeableElementKnob(container.find('.dec'),
       function () {
           if (dec.attr('disabled') === undefined) {
               var ele = dec.find('.decs')[0];
               var cName = ele.className;
               var sprite = cName.match(/\bsprite_\w*/i);
               if (sprite !== null && sprite[0].match("_pushed") === null) {
                   var spriteNew = sprite[0] + "_pushed";
                   ele.className = cName.replace(sprite[0], spriteNew);
                   dec.addClass('down');
               }
               _this.startLongPress(function () {
                   _this.decPercent(container, inc, dec, tag);
               });
           }
       },
       function () {
           if (dec.attr('disabled') === undefined) {
               var ele = dec.find('.decs')[0];
               var cName = ele.className;
               var sprite = cName.match(/\bsprite_\w*_pushed/i);
               if (sprite !== null) {
                   var spriteNew = sprite[0].replace("_pushed", "");
                   ele.className = cName.replace(sprite[0], spriteNew);
                   dec.removeClass('down');
               }
               _this.releaseLongPress();
           }
       },
       function () {
           if (dec.attr('disabled') === undefined) {
               if (!longPressInterval) {
                   _this.decPercent(container, inc, dec, tag);
               }
           }
       }
   );

    // Events on switch button
    registerEventsForSwipeableElement(container.find('.dimbutton-touch'),
       function () {
           var ele = button[0];
           var cName = ele.className;
           var sprite = cName.match(/\bsprite_\w*/i);
           if (sprite !== null && sprite[0].match("_down") === null) {
               var spriteNew = sprite[0] + "_down";
               ele.className = cName.replace(sprite[0], spriteNew);
           }
           container.find('.dec').removeClass('shadow');
           container.find('.inc').removeClass('shadow');
       },
       function () {
           var ele = button[0];
           var cName = ele.className;
           var sprite = cName.match(/\bsprite_\w*_down/i);
           if (sprite !== null) {
               var spriteNew = sprite[0].replace("_down", "");
               ele.className = cName.replace(sprite[0], spriteNew);
           }
           container.find('.dec').addClass('shadow');
           container.find('.inc').addClass('shadow');
       },
       function () {
           var onoff = knobs.val() > _this._min ? 0 : 1;
           ComClient.sendTagValue(onofftag, onoff);
           _this.setInputValue(onoff * 100);
       }
   );
};

/**
 * Sets the input value and triggers the according animation. The processing of 
 * incoming tag values is blocked for a specified amount of time.
 * @param {Integer} newValue of jquery knob as integer
 */
DimmerAbsoluteController.prototype.setInputValue = function (newValue) {
    this._inUse = true;
    this.setValueAnimated(newValue);
    var _this = this;
    clearTimeout(this._inUseTimeout);
    this._inUseTimeout = setTimeout(function () {
        _this._inUse = false;
    }, this._inUseTimer);
};

/**
 * Updates the value of the knob input and triggers an update of the canvas element if
 * the dimmer is not blocked by user interaction.
 * @param {Integer} value of jquery knob as integer
 */
DimmerAbsoluteController.prototype.update = function (value) {
    if (!(this._inUse || this.knobInstance.$c.data("active") === true)) {
        this.setValue(value);
    }
};

/**
* Updates the value text and triggers a value change animation.
* @param {Integer} value of jquery knob as integer
*/
DimmerAbsoluteController.prototype.setValueAnimated = function (value) {
    value = this.checkValue(value);
    this.setPercent(value);
    this._knobs.val(value).trigger('change');
};

/**
 * Sets the value text and triggers a value change without animation.
 * @param {Integer} value of jquery knob as integer
 */
DimmerAbsoluteController.prototype.setValue = function (value)
{
    if (!isValidValue(value)) {
        this.setPercent('-');
        return;
    }
    value = this.checkValue(value);
    this.setPercent(Round(value, 0));
    this._knobs.val(value);
    this.knobInstance.val(value, true, true);
};

/**
 * Converts value to an integer. NaN values are converted to 0.
 * @return {Integer} Numeric value
 */
DimmerAbsoluteController.prototype.checkValue = function (value) {
    value = Round(value, 0);
    return isNaN(value) ? 0 : value;
};