﻿/*global $, gds$, ComClient*/
/*jslint browser: true*/

window.tsx.log = (function () {
    'use strict';

    var log = {},
        defaultContext = "Gira.G1::Logging",
        defaultFilename = "Gira.G1";

    /**
    * writes a GDS log entry
    *
    * @param {string} msg error message
    * @param {string} logLevel gds log level
    * @param {string} context the error's context, if not profided defaultContext is used
    * @param {string} filename, if not profided defaultFilename is used
*/
    log.writeLog = function (msg, logLevel, context, filename) {
        gds$.writeLog(context ? context : defaultContext,
                      filename ? filename : defaultFilename,
                      logLevel,
                      msg);
    };

    /**
    * array for caching all errors that can't be written to the gds log by 
    * the private onerror function, will be emptied by log.writeOnGDSConnection()
    */
    log.delayedLogs = [];

    /**
    * @returns true if the local storage item "DEBUG" is set to "true", otherwise false
    */
    log.getDebugMode = function () {
        return localStorage.getItem('DEBUG') === "true";
    };

    /**
    * set DEBUG status flag in the local storage
    *
    * @param {bool} new DEBUG status
    */
    log.setDebugMode = function (bool) {
        localStorage.setItem('DEBUG', bool);
    };

    /**
    * writes a info log message
    *
    * @param {string} msg error message
    * @param {string} context the error's context, optinal
    * @param {string} filename, optinal
    */
    log.info = function (msg, context, filename) {
        log.writeLog(msg, "info", context, filename);
    };

    /**
    * writes a warning log message
    *
    * @param {string} msg error message
    * @param {string} context the error's context, optinal
    * @param {string} filename, optinal
    */
    log.warning = function (msg, context, filename) {
        log.writeLog(msg, "warn", context, filename);
    };

    /**
    * writes a error log message
    *
    * @param {string} msg error message
    * @param {string} context the error's context, optinal
    * @param {string} filename, optinal
    */
    log.error = function (msg, context, filename) {
        log.writeLog(msg, "error", context, filename);
    };

    /**
    * Callback for the Comclient Connected event
    *
    * writes all previous captured errors to the gds log
    */
    log.writeOnGDSConnection = function () {
        var f;
        for (f in log.delayedLogs) {
            log.delayedLogs[f].call();
        }

        //delete all cached error messages
        while (log.delayedLogs.length) { log.delayedLogs.pop(); }
    };

    /**
    * caches a log write function for logging as soon as the GDS connection is available 
    *
    * @param {string} msg error message
    */
    log.writeDelayed = function (logFunction) {
        log.delayedLogs.push(logFunction);
    };

    /**
    * onerror function without delayed logging on gds connection
    var onerror = function (errorMsg, url, lineNumber, column, obj) {
        if (gds$.isConnected() && !log.getDebugMode()) {
            gds$.writeLog("window.onerror", url, "error", obj.stack);
            return true;
        } else {
            return false;
        }
    };
    */

    /**
    * window.onerror function 
    * loggs all errors captured by window.onerror to the gds log file
    * if the gds connection is not available errors are cached and written as soon as the connection is established
    *
    */
    var onerror = function (errorMsg, url, lineNumber, column, obj) {

        var suppressExceptions = !log.getDebugMode();

        if (gds$.isConnected()) {
            gds$.writeLog("window.onerror", url, "error", obj.stack);
        } else {
            log.writeDelayed(function () { gds$.writeLog("window.onerror", url, "error", obj.stack); });
        }

        return suppressExceptions;
    };

    // register callback for delayed logging
    ComClient.registerOnConnectedEX(log.writeOnGDSConnection);

    //set window.onerror function
    window.onerror = onerror;
    return log;

}());

//make the log object global
log = window.tsx.log;