var fsm = (function () {
    /**
    * Request current state of actor's fsm.
    * 
    * @name {string} actors name like 'Keyboard'        
    * return: js object also used in tests. Example: { actor: 'Keyboard', current: 'off' }
    */
    function monitor(name) {
        var value = { actor: undefined, current: undefined, callbackMap: undefined };
        if (unit[actor[name]]) {
            value.actor = name;
            value.current = unit[actor[name]].stateMachine.current;
            value.callbackMap = unit[actor[name]].callbackMap;
        }
        return value;
    }
    /**
    * Invokes delegate callback
    * 
    * @callbackArgs {[function, args]} callbackArgs[0] is the callback function while callbackArgs[1] is it's the optional javascript object argument.
    */
    function call(callbackArgs) {
        if (callbackArgs && callbackArgs.length > 0) {
            var func = callbackArgs[0];
            var args = callbackArgs[1];
            if (typeof func == "function") {
                return func(args);
            }
        }
        return true;
    }

    /**
    * This function determines if the view is changed back to home-screen after screen timeout and jumps back if neccessary.
    */
    function jumpHomeAfterScreenTimeout() {
        var returnToHome = true;
        if(isState(actor.SystemMessage, 'poppedUp')) {
            returnToHome = false;
        }
        if (isState(actor.Overlay, 'on')) {
            returnToHome = false;
        }
        if (isState(actor.Blocker, 'on')) {
            returnToHome = false;
        }
        if (isState(actor.Keyboard, 'on')) {
            module.API.Keyboard.Close();
            returnToHome = true;            
        }
        if (returnToHome === true) {
            NavbarController.getInstance().openFirstFunction();
        }
    }

    /**
    * Requests actor's state machine to process an event while passing an array of length 2, containing the corresponding delegate callback
    * and it's arguments as a single java script object.
    * 
    * @key    {string} actor's name like 'Keyboard'
    * @method {string} actor's state machine event
    * @args   {string or javascript object} optional arguments
    * return true if successfull
    */
    function invoke(key, method, args) {
        if (unit[key] && unit[key].stateMachine.can([method])) {
            switch (unit[key].stateMachine[method]([unit[key].callbackMap[method], args])) {
                case StateMachine.Result.SUCCEEDED:
                case StateMachine.Result.NOTRANSITION:
                case StateMachine.Result.PENDING:
                    return true;
                case StateMachine.Result.CANCELLED:
                default:
                    return false;
            }
        }
        return false;
    }
    /**
    * Verifies actor's state
    * 
    * @name  {string} actor's name like 'Keyboard'
    * @state {string} expected current state
    * return true for the right guess
    */
    function isState(name, state) {
        return unit[name] && unit[name].stateMachine.is(state);
    }
    /**
    * Registers actor units.
    * 
    * @name         {string} actor's name like 'Keyboard'
    * @stateMachine {StateMachine} returned by StateMachine.create()
    * @callbackMap  {{string:function, ...}} map of delegate methods with actor's public methods as keys being passed to trough after triggers 'callbackArgs'.
    */
    function addUnit(name, stateMachine, callbackMap) {
        unit[name] = { key: name, stateMachine: stateMachine, callbackMap: callbackMap };
    }
    var unit = []; // contains actor units
    var actor = { 'Keyboard': 0, 'SystemMessage': 1, 'Overlay': 2, 'PrimaryFunction': 3, 'TK': 4, 'Blocker': 5, 'SystemMenu': 6 }
    var module = {
        API: {
            /**
            * Tells the State Machine to trigger the transition. Used in asynchronous state transitions.
            * 
            * @name  {string} actor's name like 'Keyboard'
            * return true if successfull
            */
            transition: function (name) {
                if (unit[actor[name]] && typeof unit[actor[name]].stateMachine.transition == "function") {
                    unit[actor[name]].stateMachine.transition();
                    return true;
                } else {
                    return false;
                }
            },
            /**
            * Tells the State Machine to cancel the async event. Used in asynchronous state transitions.
            * 
            * @name  {string} actor's name like 'Keyboard'
            * return true if successfull
            */
            transitionCancel: function (name) {
                if (unit[actor[name]] && unit[actor[name]].stateMachine.transition && typeof unit[actor[name]].stateMachine.transition.cancel == "function") {
                    unit[actor[name]].stateMachine.transition.cancel();
                    return true;
                } else {
                    return false;
                }
            },
            Keyboard: {
                monitor: function () { return monitor('Keyboard'); },
                Init: function (openCallback, closeCallback, lockCallback, unlockCallback) {
                    addUnit(actor.Keyboard, StateMachine.create({
                        initial: 'off',
                        events: [
                            { name: 'Open', from: 'off', to: 'on' },
                            { name: 'Close', from: 'on', to: 'off' },
                            { name: 'Lock', from: 'on', to: 'lock' },
                            { name: 'Unlock', from: 'lock', to: 'on' }
                        ],
                        callbacks: {
                            onbeforeevent: function (event, from, to, callbackArgs) {
                                if (event == 'Open' &&
                                    (isState(actor.SystemMessage, 'poppedUp') || isState(actor.Overlay, 'on') || isState(actor.PrimaryFunction, 'on') || isState(actor.Blocker, 'on'))) {
                                    return false;
                                }
                                call(callbackArgs);
                                return true;
                            }
                        }
                    }), { 'Open': openCallback, 'Close': closeCallback, 'Lock': lockCallback, 'Unlock': unlockCallback });
                },
                Open: function () {
                    return invoke(actor.Keyboard, 'Open');
                },
                Close: function () {
                    return invoke(actor.Keyboard, 'Close');
                },
                Lock: function () {
                    return invoke(actor.Keyboard, 'Lock');
                },
                Unlock: function () {
                    return invoke(actor.Keyboard, 'Unlock');
                }
            },
            SystemMessage: {
                monitor: function () { return monitor('SystemMessage'); },
                Init: function (openCallback, closeCallback, minimizeCallback, maximizeCallback) {
                    addUnit(actor.SystemMessage, StateMachine.create({
                        initial: 'closed',
                        events: [
                            { name: 'Open', from: '*', to: 'poppedUp' },
                            { name: 'Minimize', from: 'poppedUp', to: 'closed' },
                            { name: 'Maximize', from: 'closed', to: 'poppedUp' },
                            { name: 'Close', from: '*', to: 'closed' }
                        ],
                        callbacks: {
                            onbeforeOpen: function (event, from, to, callbackArgs) {
                                if (!isState(actor.Blocker, 'on')) {
                                    /* Attention: callback called in before trigger to change the state conditionally.
                                       
                                       Cause:
                                       The 'Open' event will pop messages up depending on its channel parameter.
                                       so if the channel has already an active message, the next call will be ignored.
                                       In that case the callback function decides whether we change the state to 'popedup' or stay in the old state.
                                    */
                                    return call(callbackArgs);
                                }
                                return false;
                            },
                            onbeforeClose: function (event, from, to, callbackArgs) {
                                /* Attention: callback called in before trigger to change the state conditionally.
                                   
                                   Cause:
                                   The 'Close' event will remove messages depending on its parameter (if set: only the given channel; otherwise: all) 
                                   so there might still be active messages from other channels left.
                                   In that case the callback function decides whether we change the state to 'closed' or stay in the old state.
                                */
                                return call(callbackArgs);
                            },
                            onbeforeMaximize: function (event, from, to, callbackArgs) {
                                return call(callbackArgs);
                            },
                            onafterevent: function (event, from, to, callbackArgs) {
                                if (event == 'Open' || event == 'Maximize') {
                                    if (isState(actor.Keyboard, 'on')) {
                                        module.API.Keyboard.Lock();
                                    }
                                } else if (event == 'Close' || event == 'Minimize') {
                                    if (isState(actor.Keyboard, 'lock')) {
                                        module.API.Keyboard.Unlock();
                                    }
                                }
                                if (event == 'Minimize') {
                                    call(callbackArgs);
                                }
                            }
                        }
                    }), { 'Open': openCallback, 'Close': closeCallback, 'Minimize': minimizeCallback, 'Maximize': maximizeCallback });
                },
                OpenError: function (itemId, message) {
                    return invoke(actor.SystemMessage, 'Open', { itemId: itemId, message: message, messageType: "error" });
                },
                OpenOkay: function (message) {
                    return invoke(actor.SystemMessage, 'Open', { message: message, messageType: "okay" });
                },
                OpenWarning: function (message) {
                    return invoke(actor.SystemMessage, 'Open', { message: message, messageType: "warning" });
                },
                OpenConfirmation: function (message, rightBtnCallback, leftBtnCallback) {
                    return invoke(actor.SystemMessage, 'Open', { message: message, rightBtnCallback: rightBtnCallback, leftBtnCallback: leftBtnCallback, messageType: "confirmation" });
                },
                Maximize: function () {
                    return invoke(actor.SystemMessage, 'Maximize');
                },
                Minimize: function (animationEndCallback) {
                    return invoke(actor.SystemMessage, 'Minimize', animationEndCallback);
                },
                CloseAll: function () {
                    return invoke(actor.SystemMessage, 'Close');
                },
                CloseError: function (itemId, animationEndCallback) {
                    return invoke(actor.SystemMessage, 'Close', { itemId: itemId, messageType: "error", animationEndCallback: animationEndCallback });
                },
                CloseOkay: function (animationEndCallback) {
                    return invoke(actor.SystemMessage, 'Close', { messageType: "okay", animationEndCallback: animationEndCallback });
                },
                CloseWarning: function (animationEndCallback) {
                    return invoke(actor.SystemMessage, 'Close', { messageType: "warning", animationEndCallback: animationEndCallback });
                },
                CloseConfirmation: function (animationEndCallback) {
                    return invoke(actor.SystemMessage, 'Close', { messageType: "confirmation", animationEndCallback: animationEndCallback });
                },
                CloseSuperError: function (animationEndCallback) {
                    return invoke(actor.SystemMessage, 'Close', { messageType: "superError", animationEndCallback: animationEndCallback });
                },
                OpenSuperError: function (message, animationEndCallback) {
                    return invoke(actor.SystemMessage, 'Open', { message: message, messageType: "superError", animationEndCallback: animationEndCallback });
                }
            },
            Overlay: {
                monitor: function () { return monitor('Overlay'); },
                Init: function (openCallback, closeCallback, openBlackCallback, closeBlackCallback, closeAllCallback) {
                    addUnit(actor.Overlay, StateMachine.create({
                        initial: 'off',
                        events: [
                            { name: 'Open', from: ['off', 'on'], to: 'on' },
                            { name: 'Close', from: 'on', to: 'off' },
                            { name: 'CloseAll', from: 'on', to: 'off' },
                            { name: 'OpenBlack', from: 'off', to: 'onBlack' },
                            { name: 'CloseBlack', from: 'onBlack', to: 'off' }
                        ],
                        callbacks: {
                            onbeforeevent: function (event, from, to) {
                                switch (event) {
                                    case 'OpenBlack':
                                        jumpHomeAfterScreenTimeout();
                                    case 'Open':
                                        var state = isState(actor.Blocker, 'on');
                                        if (isState(actor.Keyboard, 'on') && !state) {
                                            module.API.Keyboard.Lock();
                                        }
                                        return !state;
                                    default:
                                        return true;
                                }
                            },
                            onafterevent: function (event, from, to, callbackArgs) {
                                switch (event) {
                                    case 'Open':
                                    case 'OpenBlack':
                                        if (isState(actor.PrimaryFunction, 'on')) {
                                            module.API.PrimaryFunction.Close();
                                        }
                                        break;
                                    case 'Close':
                                    case 'CloseAll':
                                    case 'CloseBlack':
                                        if (isState(actor.Keyboard, 'lock') &&
                                        (!isState(actor.SystemMessage, 'poppedUp'))) {
                                            module.API.Keyboard.Unlock();
                                        }
                                    default:
                                        break;
                                }

                                call(callbackArgs);
                            }
                        }
                    }), { 'Open': openCallback, 'Close': closeCallback, 'OpenBlack': openBlackCallback, 'CloseBlack': closeBlackCallback, 'CloseAll': closeAllCallback });
                },
                Open: function (messageOne, messageTwo, hasSpinner, hasCancelButton, func, jso, countdown) {
                    return invoke(actor.Overlay, 'Open', { messageOne: messageOne, messageTwo: messageTwo, hasSpinner: hasSpinner, hasCancelButton: hasCancelButton, func: func, jso: jso, countdown: countdown });
                },
                CloseAll: function () {
                    return invoke(actor.Overlay, 'CloseAll');
                },
                Close: function (args) {
                    return invoke(actor.Overlay, 'Close', args);
                },
                OpenBlack: function () {
                    return invoke(actor.Overlay, 'OpenBlack');
                },
                CloseBlack: function () {
                    return invoke(actor.Overlay, 'CloseBlack');
                }
            },
            PrimaryFunction: {
                monitor: function () { return monitor('PrimaryFunction'); },
                Init: function (disableCallBack, openCallback, closeCallback) {
                    addUnit(actor.PrimaryFunction, StateMachine.create({
                        initial: 'off',
                        events: [
                            { name: 'Enable', from: ['off', 'standby'], to: 'standby' },
                            { name: 'Disable', from: ['standby', 'off'], to: 'off' },
                            { name: 'Open', from: 'standby', to: 'on' },
                            { name: 'Close', from: 'on', to: 'standby' }
                        ],
                        callbacks: {
                            onbeforeOpen: function (event, from, to) {
                                return !(isState(actor.Overlay, 'on') || isState(actor.Blocker, 'on') || $('#overlay').size() > 0);
                            },
                            onafterevent: function (event, from, to, callbackArgs) {
                                call(callbackArgs);
                            }
                        }
                    }), { 'Disable': disableCallBack, 'Open': openCallback, 'Close': closeCallback });
                },
                Enable: function () {
                    return invoke(actor.PrimaryFunction, 'Enable');
                },
                Disable: function () {
                    return invoke(actor.PrimaryFunction, 'Disable');
                },
                Open: function () {
                    return invoke(actor.PrimaryFunction, 'Open');
                },
                Close: function () {
                    return invoke(actor.PrimaryFunction, 'Close');
                }
            },
            TK: {
                monitor: function () { return monitor('TK'); },
                Init: function (openCallback, closeCallback) {
                    addUnit(actor.TK, StateMachine.create({
                        initial: 'off',
                        events: [
                            { name: 'Init', from: 'off', to: 'standby' },
                            { name: 'Disable', from: 'standby', to: 'off' },
                            { name: 'Open', from: 'standby', to: 'on' },
                            { name: 'Close', from: 'on', to: 'standby' }
                        ],
                        callbacks: {
                            onbeforeOpen: function (event, from, to) { return !isState('Blocker', 'on'); },
                            onafterOpen: function (event, from, to, callbackArgs) {
                                if (isState(actor.Keyboard, 'on')) {
                                    module.API.Keyboard.Close();
                                }
                                if (isState(actor.SystemMessage, 'poppedUp')) {
                                    module.API.SystemMessage.Minimize();
                                }
                                if (isState(actor.PrimaryFunction, 'on')) {
                                    module.API.PrimaryFunction.Close();
                                }
                                if (isState(actor.SystemMenu, 'poppedUp')) {
                                    module.API.SystemMenu.HardClose();
                                }

                                call(callbackArgs);
                            },
                            onafterClose: function (event, from, to, callbackArgs) {
                                call(callbackArgs);
                            }
                        }
                    }), { 'Open': openCallback, 'Close': closeCallback });

                    return invoke(actor.TK, 'Init');
                },
                Open: function () {
                    return invoke(actor.TK, 'Open');
                },
                Close: function () {
                    return invoke(actor.TK, 'Close');
                }
            },
            Blocker: {
                monitor: function () { return monitor('TK'); },
                Init: function (openCallback, closeCallback) {
                    addUnit(actor.Blocker, StateMachine.create({
                        initial: 'off',
                        events: [
                            { name: 'Open', from: 'off', to: 'on' },
                            { name: 'Close', from: 'on', to: 'off' }
                        ],
                        callbacks: {
                            onafterOpen: function (event, from, to, callbackArgs) {
                                if (isState(actor.Keyboard, 'on')) {
                                    module.API.Keyboard.Close();
                                }
                                if (isState(actor.SystemMessage, 'poppedUp')) {
                                    module.API.SystemMessage.Close();
                                }
                                if (isState(actor.Overlay, 'on')) {
                                    module.API.Overlay.CloseAll();
                                }
                                if (isState(actor.PrimaryFunction, 'on')) {
                                    module.API.PrimaryFunction.Close();
                                }
                                if (isState(actor.TK, 'on')) {
                                    module.API.TK.Close();
                                }

                                call(callbackArgs);
                            },
                            onafterClose: function (event, from, to, callbackArgs) {
                                call(callbackArgs);
                            }
                        }
                    }), { 'Open': openCallback, 'Close': closeCallback });
                },
                Open: function () {
                    return invoke(actor.Blocker, 'Open');
                },
                Close: function () {
                    return invoke(actor.Blocker, 'Close');
                }
            },
            SystemMenu: {
                monitor: function () { return monitor('SystemMenu'); },
                Init: function (openCallback, closeCallback) {
                    addUnit(actor.SystemMenu, StateMachine.create({
                        initial: 'closed',
                        events: [
                            { name: 'Open', from: 'closed', to: 'poppedUp' },
                            { name: 'Close', from: 'poppedUp', to: 'closed' },
                            { name: 'HardClose', from: 'poppedUp', to: 'closed' }
                        ],
                        callbacks: {
                            onbeforeOpen: function (event, from, to) {
                                var state = isState(actor.Popup, 'on') || isState(actor.SystemMessage, 'poppedUp')
                                    || isState(actor.Overlay, 'on') || isState(actor.PrimaryFunction, 'on')
                                    || isState(actor.Blocker, 'on');
                                return !state;
                            },
                            onafterevent: function (event, from, to, callbackArgs) {
                                call(callbackArgs);
                            }
                        }
                    }), { 'Open': openCallback, 'Close': closeCallback });
                },
                Open: function () {
                    return invoke(actor.SystemMenu, 'Open');
                },
                Close: function () {
                    return invoke(actor.SystemMenu, 'Close');
                },
                HardClose: function () {
                    return invoke(actor.SystemMenu, 'Close', { hardClose: true });
                }
            }
        }
    };
    return module;
})();