/*jslint browser: true*/
/*global console, Gira*/
(function (ns) {
    'use strict';

    if (ns.DcsToGdsConnector !== undefined) {
        return;
    }

    var log = window.Gira.Log;

    function isMessageJustNoise(message) {
        return message.indexOf(".Peripherals:") > -1;
    }

    // Class for communication between application and DCS via proxy.
    function DcsToGdsConnector(applicationId, allowPing, connection) {
        var pingInterval = 15000,
            self = this;
        this.__ws = connection;
        this.__applicationId = applicationId;
        this.__readyHandlers = [];
        this.isOpen = false;
        this.__pingIntervlId = null;

        window.clearInterval(this.__pingIntervlId);

        if (allowPing) {
            self.__pingIntervlId = window.setInterval(function () {
                self.__ping();
            }, pingInterval);
        }

        connection.onopen(function (ea) {
            self.__onOpen(ea);
        });
        connection.onclose(function (ea) {
            self.__onClose(ea);
        });
        connection.onerror(function (ea) {
            self.__onError(ea);
        });
        connection.onmessage(function (ea) {
            self.__onMessage(ea);
        });
    }

    DcsToGdsConnector.prototype = {
        constructor: DcsToGdsConnector,

        __ping: function () {
            var self = this;
            self.sendMessage({"ping": "true"});
        },
        __waitForSocketConnection: function (counter, socket, callback) {
            if (this.__ws.readyState() === 1) { // be eager
                callback();
                return;
            }

            var self = this;
            counter = counter + 1;
            if (counter > 200) {
                return;
            }
            setTimeout(
                function () {
                    if (socket.readyState() === 1) {
                        if (callback !== null) {
                            callback();
                        }
                    } else {
                        self.__waitForSocketConnection(counter, socket, callback);
                    }
                },
                10
            ); // wait 10 milliseconds for the connection...
        },
        __onOpen: function (e) {
            this.isOpen = true;
            this.__initializateConnectionProtocol();
            this.__readyHandlers.forEach(function (item) {
                item.onOpen(e);
            });
        },
        __onClose: function (e) {
            var self = this;
            this.isOpen = false;
            this.__readyHandlers.forEach(function (item) {
                item.onClose(e);
            });
            setTimeout(function () {
                self.__ws.renew();
            }, 10000);
        },
        __onError: function (e) {
            this.__readyHandlers.forEach(function (item) {
                item.onError(e);
            });
        },
        __onMessage: function (e) {
            if (e.data) {
                try {
                    if (!isMessageJustNoise(e.data)) {
                        log.info("Receive message: " + e.data);
                    }
                    if (this.__allowPing) {
                        clearTimeout(this.__pingTimeoutId);
                    }
                    var obj = JSON.parse(e.data);
                    if (this.__checkIfMessageObjectAreValid(obj)) {
                        if (!this.__checkIfIsSystemMessage(obj)) {
                            this.__readyHandlers.forEach(function (item) {
                                item.onMessage(obj);
                            });
                        }
                    }
                } catch (err) {
                    log.error("message threw while being handled: " + e.data + " " + err);
                }
            }
        },
        __checkIfMessageObjectAreValid: function (obj) {
            if (obj) {
                if (obj.response || obj.event) {
                    return true;
                }
                log.error("Invalid object: " + obj.toString());
            } else {
                log.error("Invalid object");
            }

            return false;
        },
        __initializateConnectionProtocol: function () {
            this.__sendRegisterApplication();
        },
        __registrationMessage: function (applicationId) {
            return {
                request: {
                    command: "RegisterApplication",
                    applicationId: applicationId,
                    pretty: "true"
                }
            };
        },
        // Register application with GDS API
        __sendRegisterApplication: function () {
            this.sendMessage(this.__registrationMessage(this.__applicationId));
        },
        // System messages are "SetConfiguration" and "RegisterApplication"
        __checkIfIsSystemMessage: function (obj) {
            var response = obj.response,
                request;
            if (response && response.request) {
                request = response.request;
                switch (request.command) {
                case "RegisterUI":
                case "RegisterApplication":
                    return true;
                }
            }

            return false;
        },
        getApplicationId: function () { return this.__applicationId; },
        registerListener: function (listener) {
            this.__readyHandlers.push(listener);
            if (this.isOpen) {
                listener.onOpen();
            }
        },
        unregisterListener: function (listenerType) {
            var indexForDelete = -1;
            this.__readyHandlers.forEach(function (item, index) {
                if (item instanceof listenerType) {
                    indexForDelete = index;
                    return;
                }
            });
            if (indexForDelete > -1) {
                this.__readyHandlers.splice(indexForDelete, 1);
            }
        },
        waitForSocketConnection: function (socket, callback) {
            var self = this,
                counter = 0;
            self.__waitForSocketConnection(counter, socket, callback);
        },
        close: function () {
            var self = this;
            if (self.__ws.readyState() === 1) {
                self.__ws.close();
            }
        },
        sendMessage: function (message) {
            try {
                var messageAsString = JSON.stringify(message),
                    self = this;
                log.info("Send message: " + messageAsString);
                self.rawSend(messageAsString);
            } catch (err) {
                log.error("exception thrown while sending message: " + message + " " + err);
            }
        },
        rawSend: function (messageAsString) {
            var self = this;
            if (self.__ws.readyState() === 3) {
                self.__ws.renew();
            }
            // Wait until the state of the socket is ready and send the message when it is...
            self.waitForSocketConnection(self.__ws, function () {
                self.__ws.send(messageAsString);
            });
        }
    };

    ns.DcsToGdsConnector = DcsToGdsConnector;

}(window.Gira));

//# sourceURL=gira.gdsConnector.js