/**
 * A Case is a comparison based selection for a single return value
 */
function Case()
{
    this._upperLimitValue = null;
    this._lowerLimitValue = null;
    this._upperOperator = null;
    this._lowerOperator = null;
    this._newValue = null;
    this._expectedValue = null;
}
;

/**
 * Checks if input value matches for this case
 * 
 * @param input value * 
 * @return true on success, else false
 */
Case.prototype.matchesInput = function(input) {
    if (this._expectedValue !== null) {
        if (input === this._expectedValue) {
            return true;
        }
    } else if (this._matchesLowerEquationPart(input) && this._matchesUpperEquationPart(input)) {
        return true;
    }
    return false;
}
;

/**
 * Returns the value associated with this case
 * 
 * @return associated value
 */
Case.prototype.getNewValue = function() {
    return this._newValue;
}
;

/**
 * Checks if input is within upper equation part
 * 
 * @param input value
 * @return true on success, else false
 * @private
 */
Case.prototype._matchesUpperEquationPart = function(input) {
    if (this._upperOperator === CaseOperator.LESS) {
        if (input < this._upperLimitValue) {
            return true;
        }
    }
    if (this._upperOperator === CaseOperator.LESS_OR_EQUAL) {
        if (input <= this._upperLimitValue) {
            return true;
        }
    }
    return false;
}
;

/**
 * Checks if input is within lower equation part
 * 
 * @param input value
 * @return true on success, else false
 * @private
 */
Case.prototype._matchesLowerEquationPart = function(input) {
    if (this._lowerOperator === CaseOperator.GREATER) {
        if (input > this._lowerLimitValue) {
            return true;
        }
    } else if (this._lowerOperator === CaseOperator.GREATER_OR_EQUAL) {
        if (input >= this._lowerLimitValue) {
            return true;
        }
    }
    return false;
}
;

/* CaseBuilder start*/

/**
 * Sets the limits for the comparison
 * 
 * @param lowerLimitValue
 * @param upperLimitValue
 * @return the current instance
 */
Case.prototype.setLimits = function(lowerLimitValue, upperLimitValue) {
    if (this._expectedValue !== null) {
        throw "setLimits can't be used with expectedValue; please remove setExpectedValue from your Builder-call.";
    }
    if (lowerLimitValue !== null && upperLimitValue !== null) {
        this._upperLimitValue = upperLimitValue;
        this._lowerLimitValue = lowerLimitValue;
    }
    else {
        throw "upperLimitValue or lowerLimitValue is not set";
    }

    return this;
}
;

/**
 * Sets the limit comparison {@link Operator}s
 * 
 * @param lowerOperator
 * @param upperOperator
 * @return the current instance
 * @throws IllegalArgumentException
 */
Case.prototype.setRangeOperators = function(lowerOperator, upperOperator) {
    if (this._expectedValue !== null) {
        throw "expectedValue already set in builder; can't add rangeOperators because expectedValue forces an equals-comparison. Please fix your Builder-call.";
    }
    if (lowerOperator !== CaseOperator.GREATER && lowerOperator !== CaseOperator.GREATER_OR_EQUAL) {
        throw "lowerOperator must be Operator.GREATER or Operator.GREATER_OR_EQUAL";
    }
    if (upperOperator !== CaseOperator.LESS && upperOperator !== CaseOperator.LESS_OR_EQUAL) {
        throw "upperOperator must be Operator.LESS or Operator.LESS_OR_EQUAL";
    }
    this._lowerOperator = lowerOperator;
    this._upperOperator = upperOperator;

    return this;
}
;

/**
 * Sets the value that shall be used on match
 * 
 * @param newValue value
 * @return the current instance
 */
Case.prototype.setNewValue = function(newValue) {
    this._newValue = newValue;
    return this;
}
;

/**
 * Convenience method for bypassing limit & operator setting for exact match
 * 
 * @param expectedValue value
 * @return the current instance
 */
Case.prototype.setExpectedValue = function(expectedValue) {
    if (this._lowerOperator !== null || this._upperOperator !== null) {
        throw "rangeOperators already set in Builder; expectedValue can't be used with rangeOperators, please fix your Builder-call.";
    }
    if (this._lowerLimitValue !== null || this._upperLimitValue !== null) {
        throw "setExpectedValue can't be used with value-range-limits; please remove setLimits() from your Builder-call.";
    }
    this._expectedValue = expectedValue;

    return this;
}
;

/**
 * Validate a {@link Case} based on this builder's parameters
 * 
 * @return true
 * @throws IllegalStateException
 */
Case.prototype.isValid = function() {
    var caseIsValid = false;
    /* general purpose variables have to be set always */
    if (this._newValue !== null) {
        /* value range comparison */
        if ((this._upperLimitValue !== null && this._lowerLimitValue !== null) && (this._lowerOperator !== null && this._upperOperator !== null)) {
            caseIsValid = true;
        } else {
            if (this._expectedValue !== null) {
                caseIsValid = true;
            }
        }
    }
    if (caseIsValid) {
        return true;
    } else {
        throw "Some fields are not filled, please check the CaseBuilders in your code.";
    }
}
;

/* CaseBuilder end */

/** 
 * @enum Case operator 
 */
CaseOperator = {
    GREATER: 0,
    GREATER_OR_EQUAL: 1,
    LESS: 2,
    LESS_OR_EQUAL: 3,
    NONE: 4,
    OR: 5,
    AND: 6,
    XOR: 7,
    SUM: 8
}
;