﻿/**
 *	Blind Absolute Tile Controller 
 *
 *  Create a new controller instance for a blind absolute tile
 *
 * @param {string} tileId id of tile
 * @param {string} btnDownId down button id
 * @param {string} btnUpId up button id
 * @param {string} DPHeightPosition tag of height position datapoint
 * @param {string} DPHeightPositionState tag of height position status datapoint
 *				   if zero DPHeightPosition will serve as status
 * @param {string} DPMovementState tag of movement datapoint
 *				   zero if not existing
 * @param {string} DPStop tag of stop datapoint
 * @param {int}    step step size for increment / decrementing the blind position
*/
function BlindAbsoluteTile(tileId, btnDownId, btnUpId, dpHeightPosition, dpHeightPositionState, dpMovementState, dpStop, step, statusFeedback) {
    var context = this;

    this._automatic = false;
    this._automaticFirstValue = false;
    this._autoTimeOut = false;
    this._autoTimeOutTime = 4500;
    this._statusFeedback = statusFeedback == "true" ? true : false;

    this._animating = false;
    this._animation = 0;
    this._animationHeightValue = 0;
    this._animationDirection = 0;

    //Datapoints
    this._DPHeightPosition = dpHeightPosition;
    this._DPHeightPositionState = dpHeightPositionState;
    this._DPMovementState = dpMovementState === '' ? 0 : dpMovementState;

    this._DPStop = dpStop;

    //cached dp values
    this._lastHeightValue = undefined;
    this._userValue = undefined;
    this._movementState = 0;

    this._initWithSetpoint = false;

    /** 
	* Start automatic mode 
	* 
	* @param {int} target target value to animate to
	*/
    this._startAutomatic = function (target) {
        if (this._lastHeightValue === target) {
            return;
        }

        if (this._automatic === true) {
            this._stopAutomatic();
        }

        this._automatic = true;
        this._automaticFirstValue = true;

        if (context._DPMovementState === 0) // no movement status
        {
            context._autoTimeOut = setTimeout(function () {
                context._stopAutomatic();
            }, context._autoTimeOutTime);
        }

        this._startAnimation(target);
    };

    /**
	 * stop automatic mode
	*/
    this._stopAutomatic = function () {
        window.clearTimeout(context._autoTimeOut);

        this._automatic = false;
        this._stopAnimation();
    };


    /** 
	* Start automatic mode animation
	* 
	* @param {int} target target value to animate to
	*/
    this._startAnimation = function (target) {
        if (this._animating === true) {
            this._stopAnimation();
        }

        var dv = step * (target <= 0 ? -1 : 1);
        var dt = globals.dimmer.animationSpeed * Math.abs(dv);

        this._animating = true;
        this._animation = window.setInterval(function() {
            context._animate(dv, target);
        }, dt);
    };

    /**
	 * interval animation method
	 *
	 * @param {int} animationStep    step to inc/dec value for each interval
	 * @param {int} target  target value
	*/
    this._animate = function (nextStep, target) {
        context._animationHeightValue += parseInt(nextStep);

        if (nextStep > 0 && context._animationHeightValue > target) context._animationHeightValue = target;
        if (nextStep < 0 && context._animationHeightValue < target) context._animationHeightValue = target;

        context._onHeightStatusValue(context._animationHeightValue);
        if (context._DPMovementState === 0) {
            context._userValue = context._animationHeightValue;
        }

        if (context._animationHeightValue === target) {
            context._stopAnimation();
        }
    };

    /**
	 * stop automatic mode animation
	*/
    this._stopAnimation = function () {
        clearInterval(this._animation);
        context._animating = false;
        context._onAnimationEnd();
    };

    /**
	 * callback, called after the animation has ended
	*/
    this._onAnimationEnd = function () {
        if (context._movementState === 0 && context._DPMovementState > 0) {
            context._animationHeightValue = context._lastHeightValue;
            context._onHeightStatusValue(context._lastHeightValue);
        }
        else if (!context._DPMovementState > 0) {
            context._animationHeightValue = context._lastHeightValue = context._userValue;
            context._onHeightStatusValue(context._userValue);
        }
    };

    /**set buttons status enabled or disabled depending on value
    *
    * @param {string} value bild position value
    *
    *
    */
    this.setBtnStatus = function(value) {
        if (this._statusFeedback) {
            if (value <= 0.5) {
                this._btnUp.setEnabled(false);
                this._btnDown.setEnabled(true);
            } else if (value >= 99.5) {
                this._btnUp.setEnabled(true);
                this._btnDown.setEnabled(false);
            } else {
                this._btnUp.setEnabled(true);
                this._btnDown.setEnabled(true);
            }
        } else {
            //behavior is not correctly defined by Gira: blinds / shutter without feedback might lead to inactive buttons on wrong occasions
            //we leave buttons enabled in this case
            this._btnUp.setEnabled(true);
            this._btnDown.setEnabled(true);
        }
    };


    /**
	 * callback for incoming Height value
	 *
	 * @param {string} value new value
	*/
    this._onHeightStatusValue = function (value) {
        var internalValue = value;
        var displayValue;

        if (isValidValue(parseInt(internalValue))) {
            this._lastHeightValue = Round(internalValue, 0);
            this._userValue = this._lastHeightValue;
            displayValue = this._lastHeightValue;
        } else {
            internalValue = undefined;
            this._lastHeightValue = undefined;
            this._userValue = undefined;
            displayValue = "-";
        }

        if (this._animating === false) {
            this._animationHeightValue = this._lastHeightValue;
            this._userValue = this._lastHeightValue;
        }
        else if (this._animating === true) {
            if (displayValue === '-') {
                displayValue = 0;
            }
        }

        if (this._automatic && this._automaticFirstValue) {
            this._automaticFirstValue = false;
        } else {
            changeText(new Array(tileId, 'tile-status-text'), displayValue + ' %');
        }
        this.setBtnStatus(internalValue);
    };

    /**
	 * register on status datapoint
	 * @param {string} valueStatusDatapoint 
	*/
    function registerHeightStatusValue(valueStatusDatapoint) {
        temporaryCaseSets.push(new CaseSet(valueStatusDatapoint, CaseSetType.VALUE)
			.setOnValueFunction(function (value) {
			    if (!context._initWithSetpoint) {
			        context._onHeightStatusValue(value);
			    }
			    else {
			        context._initWithSetpoint = false;
			    }
			})
		);
    }

    /**
	 * callback for incoming movement state
	 *
	 * @param {string} value new value
	*/
    this._onMovementState = function (value) {
        this._movementState = parseInt(value);
    };

    /**
	 * register on movement datapoint
	 * @param {string} valueStatusDatapoint 
	*/
    function registerMovementState(valueStatusDatapoint) {
        temporaryCaseSets.push(new CaseSet(valueStatusDatapoint, CaseSetType.VALUE)
			.setOnValueFunction(function (value) {
                var internalValue = value;
                if (!isValidValue(internalValue)) {
                    internalValue = 0;
			    }
                context._onMovementState(internalValue);
			})
		);
    }

    /**
	* register on init function
	* @param {string} positionTag Id of the position datapoint 
    * @param {string} positionTag Id of the movment status datapoint 
	*/
    function registerOnInit(positionTag, movementStateTag) {
        $('#' + tileId).on('init', function () {

            if (parseInt(ComClient.getTagValue(movementStateTag)) === 1) {
                var setpointValue = ComClient.getTagValue(positionTag);
                context._onHeightStatusValue(setpointValue);
                context._initWithSetpoint = true;
            }
        });
    }

    this._btnUp = new BlindAbsoluteButton(tileId, btnUpId, -step, context);
    this._btnDown = new BlindAbsoluteButton(tileId, btnDownId, step, context);

    registerHeightStatusValue(this._DPHeightPositionState);
    registerMovementState(this._DPMovementState);
    registerOnInit(this._DPHeightPosition, this._DPMovementState);
};