﻿var rtc = (function () {
    "use strict";
    var my = {},
        I18nTexts = window.Gira.appLang.appI18nTexts;
    my.flipped = false;
    my.lastActiveMode = "";
    my.lastActiveLiElement = "";
    my.operationModeTag = "";
    my.heatingcoolingMode = "";
    my.shiftUpTag = "";
    my.shiftDownTag = "";
    my.setpointTemperature = "";
    my.setpointTempAbsolute = "";
    my.detailId = "";
    my.rtcId = "";
    my.isThermProtActive = false;
    var optsMinTemp = 0;            // initial values assure testing; will be overwritten by gds see readSetPointTags
    var optsMaxTemp = 400;
    var rtcScaleMin = 100;          //minimum visible temperature (10.0 °C)
    var rtcScaleMax = 300;          //maximum visible temperature (30.0 °C)
    var lock = false;               //"RTC locked"-flag (prevents all user input)
    var currentKnob = null;         //knob instance
    my.longPressInterval = null;
    var longPressTimeout = null;
    var opts = {};

    /*
        data structure for RTC modes; this isn't used to it's full extend anymore and could be refactored
    */
    var defaults = {
        ctx: '#rtc',
        steps: 5,
        tpl: null,
        isTemp: undefined,
        setTemp: undefined,
        modes: {
            comfort: {
                icon: 'undefinedIcon',
                label: I18nTexts.comfortMode,
                hot: '21.0',
                cold: '21.5',
                temp: 220,
                eco: false,
                active: true,
                clicked: false,
                tagValue: 1
            },
            standby: {
                icon: 'undefinedIcon',
                label: I18nTexts.standBy,
                hot: '18.0',
                cold: '21.5',
                temp: 200,
                eco: true,
                active: false,
                clicked: false,
                tagValue: 2
            },
            economy: {
                icon: 'undefinedIcon',
                label: I18nTexts.night,
                hot: '15.0',
                cold: '21.5',
                temp: 180,
                eco: true,
                active: false,
                clicked: false,
                tagValue: 3
            },
            thermalprotection: {
                icon: 'undefinedIcon',
                label: I18nTexts.thermalProtection,
                hot: '15.0',
                cold: '21.5',
                temp: 180,
                eco: true,
                active: false,
                clicked: false,
                tagValue: 4
            }
        },
        heatingDiff: 20,
        coolingDiff: 20,
        imgGradient: new Image(),
        imgHatch: new Image()
    };

    /*
    * Main init function; currently has way too many parameters, could be refactored
    */
    my.init = function (detailId, dataset, arrowRight, hatchImage, iconPlus, iconMinus, opModes) {
        my.operationModeTag = dataset.operationModeId;
        my.operationModeStatusTag = dataset.operationModeStatusId;
        my.detailId = detailId;
        my.shiftUpTag = dataset.setpointTemperatureShiftUpId;
        my.shiftDownTag = dataset.setpointTemperatureShiftDownId;
        my.setpointTemperature = dataset.setpointTemperatureShiftStatusId;
        my.setpointTempAbsolute = dataset.setpointTemperatureShiftAbsoluteId;
        my.setpointTempStatus = dataset.setpointTemperatureStatusId;
        my.hatchImage = 'img/' + hatchImage.slice(hatchImage.indexOf('_') + 1) + '.png';
        my.rtcId = dataset.rtcId;
        opts = defaults;
        readSetPointTags();
        RtcUi.init(iconMinus, iconPlus, arrowRight);
        initKnobs();
        RtcModes.init(opModes, dataset.operationModeId, dataset.rtcId);
        setTemperatureKnobCaseSet(dataset.setpointTemperatureStatusId);
    };

    /*
    * Sets the RTC heating/cooling mode as configured in ETS
    */
    my.setheatingCoolingMode = function(heatingCoolingMode) {
        RtcModes.heatingCoolingMode = heatingCoolingMode;
        RtcModes.opts = opts;
    };

    /*
    * Initializes the RTC slider to it's current value, if available; 
    * The slider is set without animation
    */
    my.initRtcTemp = function () {
        var value = ComClient.getTagValue(my.setpointTempStatus);
        value = checkValue(value);
        currentKnob.val(value * 10, true, true);
    };

    /*
    * Returns true, if RTC is currently locked (no user interaction allowed); false otherwise
    */
    my.isRtcLocked = function () {
        return lock;
    };

    /*
    * Sets the RTC lock flag; see RtcUiController.setLockMode() for further details
    */
    my.setRtcLock = function (isLocked) {
        lock = isLocked;
    };

    /*
    * Returns the minimal temperature limit (defined by frost protection)
    */
    my.getOptsMinTemp = function () {
        return optsMinTemp;
    };

    /*
    * Returns the maximum temperature limit (defined by heat protection)
    */
    my.getOptsMaxTemp = function () {
        return optsMaxTemp;
    };

    /*
    * Registers the caseSet for Eco-Mode
    */
    my.registerRtcEcoModeCaseSet = function (ecoModusStatus) {
        var $ecoMode = $("#rtc .rtc-ecomode-img");

        temporaryCaseSets.push(new CaseSet(ecoModusStatus, CaseSetType.CONTENT)
            .addCase(new Case().setExpectedValue(1).setNewValue(true))
            .addCase(new Case().setExpectedValue(0).setNewValue(false))
            .setOnValueFunction(function (value) {
                if (value) {
                    $ecoMode.show();
                } else {
                    $ecoMode.hide();
                }
            })
        );
    };

    /*
    * Initializes the RTC mode extension / presence button; currently both buttons look and work the same, but affect 
    * different RTC modes.
    * 
    * @param {String} buttonId the button ID off the mode extension / presence button
    * @param {int} dataPoint the receiving datapoint for mode extension / presence actions
    * @param {String} the CSS image class for the button
    * @param {String} the CSS image class for the down-status of the button
    * @param {String} the CSS image class for the active status of the button
    */
    my.registerRtcModeExtensionButton = function (buttonId, dataPoint, buttonImg, buttonDownImg, buttonImgActive) {
        var modeExtContainer = $('#' + buttonId);
        var modeExtButton = $('#' + buttonId + ' .rtc-mode-extension-button');
        var modeExtImg = $('#' + buttonId + ' .rtc-mode-extension-img');

        registerEventsForSwipeableElement(modeExtContainer,
            function () {
                modeExtImg.addClass(buttonDownImg);
                modeExtImg.removeClass(buttonImg);
                modeExtButton.addClass("down");
            },
            function () {
                var isActive = modeExtButton.hasClass('active');
                modeExtImg.removeClass(buttonDownImg);
                if (isActive) {
                    modeExtImg.addClass(buttonImgActive);
                } else {
                    modeExtImg.addClass(buttonImg);
                }
                modeExtButton.removeClass("down");
            },
            function () {
                var isActive = modeExtButton.hasClass('active');
                if (isActive) {
                    ComClient.sendTagValue(dataPoint, 0);
                } else {
                    ComClient.sendTagValue(dataPoint, 1);
                }
            });
    };

    /**
    * Registers the case sets for actual temperature and heating/cooling icons
    * 
    * @param {object} dataset dataset comprising channel datapoints
    * @param {bool} isControllerOperationLock Flag for operation lock
    * @param {String} heatingIcon Sprite class of heating icon
    * @param {String} coolingIcon Sprite class of cooling icon
    */
    my.registerRtcTemperatureStatusCaseSets = function (dataset, isControllerOperationLock, heatingIcon, coolingIcon, dewPointIcon) {
        var $actualTempElement = $("#rtc .is-temp");
        var $heatingCoolingElement = $("#rtc .heating-cooling-status");

        if (isControllerOperationLock) {
            // controllerOperationLockStatus case set
            temporaryCaseSets.push(new CaseSet(dataset.controllerOperationLockStatusId, CaseSetType.VALUE)
                .setOnValueFunction(function (value) {
                    RtcUi.setLockMode(parseInt(value));
                })
            );
        }

        // Actual temperature case set
        temporaryCaseSets.push(new CaseSet(dataset.actualTemperatureId, CaseSetType.VALUE)
            .setOnValueFunction(function (value) {
                var temperature = parseFloat(value).toFixed(1);
                if (isNaN(temperature)) {
                    temperature = '- - -';
                }
                $actualTempElement.text(temperature + ' °C');
            })
        );

        // Heating status case set
        temporaryCaseSets.push(new CaseSet(dataset.heatingStatusId, CaseSetType.CONTENT)
            .addCase(new Case().setExpectedValue(1).setNewValue(true))
            .addCase(new Case().setExpectedValue(0).setNewValue(false))
            .setOnValueFunction(function (value) {
                if (value) {
                    $heatingCoolingElement.removeClass(coolingIcon).addClass(heatingIcon);
                } else {
                    $heatingCoolingElement.removeClass(heatingIcon);
                }
            })
        );

        // Cooling status case set
        temporaryCaseSets.push(new CaseSet(dataset.coolingStatusId, CaseSetType.CONTENT)
            .addCase(new Case().setExpectedValue(1).setNewValue(true))
            .addCase(new Case().setExpectedValue(0).setNewValue(false))
            .setOnValueFunction(function (value) {
                if (value) {
                    $heatingCoolingElement.removeClass(heatingIcon).addClass(coolingIcon);
                } else {
                    $heatingCoolingElement.removeClass(coolingIcon);
                }
            })
        );

        // DewPoint status case set
        temporaryCaseSets.push(new CaseSet(dataset.dewPointStatusId, CaseSetType.CONTENT)
            .addCase(new Case().setExpectedValue(1).setNewValue(true))
            .addCase(new Case().setExpectedValue(0).setNewValue(false))
            .setOnValueFunction(function (value) {
                if (value) {
                    $heatingCoolingElement.removeClass(heatingIcon).removeClass(coolingIcon).addClass(dewPointIcon);
                } else {
                    $heatingCoolingElement.removeClass(dewPointIcon);
                }
            })
        );
    };

    /*
    * Sets the lock flag (see RtcUiController.setLockMode for further details)
    *
    * @param {boolean} rtcIsLocked sets the lock status flag of the RTC; true, if RTC is locked; false otherwise
    */
    my.setLockFlag = function(rtcIsLocked) {
        my.lock = rtcIsLocked;
    };

    /*
    * Returns the lock flag (see RtcUiController.setLockMode)
    */
    my.getLockFlag = function() {
        return my.lock;
    };

    my.getAdjustedTemp = function() {
        return opts.setTemp;
    };

    my.getKnob = function() {
        return currentKnob;
    };

    /*
    * Loads the gradient image used for the slider bar
    */
    function loadGradientImg(onLoad) {
        opts.imgGradient.onload = onLoad;
        opts.imgGradient.src = 'img/gc4_rtr_skala_gradient.png';
    }

    /*
    * Loads the minMaxLimit Slider bar image (formerly known as hatch)
    */
    function loadHatchImg(onLoad) {
        opts.imgHatch.onload = onLoad;
        opts.imgHatch.src = my.hatchImage;
    }

    /*
    * Reads the datapoints from GDS and stores them into the RTC data structure
    */
    function readSetPointTags() {
        tsxGdsUtility$.getTemperatureSetPoint(my.rtcId, function (setPoints) {
            opts.modes.comfort.hot = setPoints["ComfortHeatTemperatureLimit"];
            opts.modes.comfort.cold = setPoints["ComfortCoolTemperatureLimit"];
            opts.modes.standby.hot = setPoints["StandbyHeatTemperatureLimit"];
            opts.modes.standby.cold = setPoints["StandbyCoolTemperatureLimit"];
            opts.modes.economy.hot = setPoints["EconomyHeatTemperatureLimit"];
            opts.modes.economy.cold = setPoints["EconomyCoolTemperatureLimit"];
            opts.modes.thermalprotection.hot = setPoints["FrostProtectionTemperature"];
            opts.modes.thermalprotection.cold = setPoints["HeatProtectionTemperature"];


            if (!isNaN(opts.modes.thermalprotection.cold)) {
                optsMaxTemp = opts.modes.thermalprotection.cold * 10;
            }
            if (!isNaN(opts.modes.thermalprotection.hot)) {
                optsMinTemp = opts.modes.thermalprotection.hot * 10;
            }
            RtcUi.drawMinMaxLimits(currentKnob, opts.imgHatch);
        });
    }

    /*
    * Increase the temperature by 1 step (write directly to datapoint)
    */
    function incTemp() {
        ComClient.sendTagValue(my.shiftUpTag, 1);
    }

    /*
    * Decrease the temperature by 1 step (write directly to datapoint)
    */
    function decTemp() {
        ComClient.sendTagValue(my.shiftDownTag, 1);
    }

    /*
    * Sets the requested temperature display ("Soll-Temperatur") and updates the +/- button states
    */
    function setTemp($container, temp)
    {

        if (!isValidValue(temp))
            return;

        if (temp > optsMaxTemp)
        {
            temp = optsMaxTemp;
        }
        else
        if (temp < optsMinTemp)
        {
            temp = optsMinTemp;
        }

        opts.setTemp = temp;        

        $container.find('.set-temp').text((opts.setTemp / 10).toFixed(1));

        if (!lock) {
            RtcUi.updateIncDecButtons($container, temp);
        }
    }

    my.setThermalProtectionActive = function (isActive) {
        my.isThermProtActive = isActive;
    };

    my.isThermalProtectionActive = function() {
        return my.isThermProtActive;
    }

    /*
    * Helper function to initiate long press actions
    */
    my.startLongPress = function(fnc) {
        clearTimeout(longPressTimeout);
        longPressTimeout = setTimeout(function() {
            clearInterval(my.longPressInterval);
            my.longPressInterval = setInterval(function() {
                fnc();
            }, 300);
        }, 300);
    };

    /*
    * Helper function to clear long press actions
    */
    my.releaseLongPress = function() {
        setTimeout(function() {
            clearInterval(my.longPressInterval);
            clearTimeout(longPressTimeout);
            my.longPressInterval = null;
        }, 50);
    };

    /*
    * Registers events for several UI elements like knob and RTC modes button
    */
    function bindEvents($container) {
        var element = $container.find('canvas');
        // Events on canvas (ring)
        registerEventsForPressButton(element, function() {
            element.closest('.swiper-slide').addClass('noSwiping');
        }, function() {
            element.closest('.swiper-slide').removeClass('noSwiping');
        }, function () { });

        var inc = $container.find('.inc');
        var dec = $container.find('.dec');

        registerEventsForSwipeableElementKnob(inc,
            function (e) {
                if (inc.attr('disabled') === undefined) {
                    var ele = inc.find('.incs')[0];
                    var cName = ele.className;
                    var sprite = cName.match(/\bsprite_\w*/i);
                    if (sprite !== null && sprite[0].match("_pushed") === null) {
                        var spriteNew = sprite[0] + "_pushed";
                        ele.className = cName.replace(sprite[0], spriteNew);
                        inc.addClass('down');
                    }
                    rtc.startLongPress(function () {
                        incTemp();
                    });
                }
            },
            function (e) {
                if (inc.attr('disabled') === undefined) {
                    var ele = inc.find('.incs')[0];
                    var cName = ele.className;
                    var sprite = cName.match(/\bsprite_\w*_pushed/i);
                    if (sprite !== null) {
                        var spriteNew = sprite[0].replace("_pushed", "");
                        ele.className = cName.replace(sprite[0], spriteNew);
                        inc.removeClass('down');
                    }
                    rtc.releaseLongPress();
                }
            },
            function () {
                if (inc.attr('disabled') === undefined) {
                    if (!my.longPressInterval) {
                        incTemp();
                    }
                }
                
            }
        );

        registerEventsForSwipeableElementKnob(dec,
            function (e) {
                if (dec.attr('disabled') === undefined) {
                    var ele = dec.find('.decs')[0];
                    var cName = ele.className;
                    var sprite = cName.match(/\bsprite_\w*/i);
                    if (sprite !== null && sprite[0].match("_pushed") === null) {
                        var spriteNew = sprite[0] + "_pushed";
                        ele.className = cName.replace(sprite[0], spriteNew);
                        dec.addClass('down');
                    }
                    rtc.startLongPress(function () {
                        decTemp();
                    });
                }
            },
            function (e) {
                if (dec.attr('disabled') === undefined) {
                    var ele = dec.find('.decs')[0];
                    var cName = ele.className;
                    var sprite = cName.match(/\bsprite_\w*_pushed/i);
                    if (sprite !== null) {
                        var spriteNew = sprite[0].replace("_pushed", "");
                        ele.className = cName.replace(sprite[0], spriteNew);
                        dec.removeClass('down');
                    }
                    rtc.releaseLongPress();
                }
            },
            function () {
                if (dec.attr('disabled') === undefined) {
                    if (!my.longPressInterval) {
                        decTemp();
                    }
                }
            }
        );

        var elementRtcModiButton = $container.find('.footer > .btn');
        registerEventsForPressButton(elementRtcModiButton,
           function () {
               elementRtcModiButton.addClass('down');
               RtcUi.setModeButtonArrowDown(elementRtcModiButton, true);
           }, function () {
               elementRtcModiButton.removeClass('down');
               RtcUi.setModeButtonArrowDown(elementRtcModiButton, false);
           }, function () {
               loadRtcModesView(my.detailId);
               rtc.showModi($container);
           });
    }

    /*
    * Shows the RTC Modes view (coverflip to the back side of the RTC)
    */
    my.showModi = function ($container) {
        RtcModes.initRtcModesEvents($('.rtc-controller'));
        RtcModes.buildModes();

        //get last active element end remove all active classes
        var activeLiElement = RtcModes.getActiveLiElement();
        if (activeLiElement) {
            activeLiElement.find('.radio').removeClass("active");
            activeLiElement.removeClass("active");
        }

        //set active element
        var operationMode = parseInt(ComClient.getTagValue(my.operationModeStatusTag));
        RtcModes.setModeActive(operationMode);

        my.backFunction = function () {
            hideModi($container, true);
        };

        rtc.flipped = true;
        coverflipFrontToBack('#' + my.detailId, true);
    }

    /*
    * Switches the coverflip back to front to display the RTC controls and hide the RTC modes
    */
    function hideModi($container, animation, callback) {
        if (rtc.flipped) {
            if (animation) {
                coverflipBackToFront('#' + my.detailId, true);
            }
            else {
                if (callback) {
                    callback();
                }
            }
            rtc.flipped = false;         
        }
    }   

    /*
    * Registers the caseSet for current temperature value changes
    * 
    * @param{int} datapoint for current temperature value
    */
    function setTemperatureKnobCaseSet(setpointTemperatureStatus) {
        var $cont = $("#rtc");

        // Set temperature case set
        temporaryCaseSets.push(new CaseSet(setpointTemperatureStatus, CaseSetType.VALUE)
            .setOnValueFunction(function (value)
            {
                var temperature = parseFloat(value * 10).toFixed(1);
                if (!isValidValue(temperature))
                {
                    temperature = '- - -';              
                    setTemp($cont, temperature);
                    $cont.find('.set-temp').text(temperature);
                }
                else
                {
                    setTemp($cont, temperature);
                    var displayTemp = applyTemperatureScaleLimits(temperature);
                    $cont.find('.knob').val(displayTemp).trigger('change');
                }
            })
        );
    }

    /*
    * Helper function to limit a given temp to both scale and frost/heat protection
    * (for display purposes only)
    */
    function applyTemperatureScaleLimits(temp) {
        var retval = temp;

        retval = retval <= optsMinTemp ? optsMinTemp : retval;
        retval = retval <= rtcScaleMin ? rtcScaleMin : retval;

        retval = retval >= optsMaxTemp ? optsMaxTemp : retval;
        retval = retval >= rtcScaleMax ? rtcScaleMax : retval;

        return retval;
    }
    
    /*
    * Initializes the Knob slider
    */
    function initKnobs() {
        $(opts.ctx + ' .rtc-controller').each(function () {
            loadGradientImg(function () {
                //nothing to do
            });
            loadHatchImg(function () {
                RtcUi.drawMinMaxLimits(currentKnob, opts.imgHatch);
            });

            var $container = $(this);
            var $knob = $container.find('.knob');

            currentKnob = $knob.knob({
                width: $knob.width(),
                height: $knob.height(),
                setpoint: opts.imgGradient,
                animationSpeed: 8,
                change: function (val) {
                    var correctedVal = applyTemperatureScaleLimits(val);
                    setTemp($('#rtc'), correctedVal);
                    return correctedVal;
                },
                sendValue: function (value) {
                    ComClient.sendTagValue(my.setpointTempAbsolute, value / 10.0);
                    $knob.val(value, true, true);
                }
            });

            setTemp($container, opts.setTemp);
            bindEvents($container);
        });
    }

    /*
    * Helper function to assert a valid integer value
    */
    function checkValue(value) {
        value = Round(value, 0);
        return isNaN(value) ? 0 : value;
    }

    /*
    * Helper function to load the RTC modes view to the back side of the RTC (coverflip animation)
    */
    function loadRtcModesView(id) {
        loadView('templates/rtc_modes.xhtml',
            function (data) {
                $('#' + id).find('.coverflip-viewsides').append(data);
            });
    }

    return my; 

})();











