
/**
 * Constructor
 * create a numeric string input controller
 * 
 * @param {type} initialValue
 *                  initial value to show
 * @param {type} maxChars
 *                  max number of characters to show
 * @param {type} isMasked
 *                  true: output is masked with '*'
 * @param {type} signEnabled                  
 *                  true: the sign change button is not disabled
 * @param {type} floatEnabled
 *                  true: the decimalPoint button is not disabled
 *                  
 * @returns {NumericInputController}
 */
function NumericInputController(maxChars, isMasked, initalValue, symbol, signEnabled, floatEnabled)
{
   this._value = initalValue;   
   this._display;
   this._maxChars = maxChars;
   this._isMasked = isMasked;
   this._symbol = symbol;
   this._onValueChangedCallBack = function(){};
   this._signEnabled = signEnabled;
   this._floatEnabled = floatEnabled;
   
   
   // map key to digit
   this._keyMap = {"btnNumOne"    : "1",
                   "btnNumTwo"    : "2",
                   "btnNumThree"  : "3",
                   "btnNumFour"   : "4",
                   "btnNumFive"   : "5",
                   "btnNumSix"    : "6",
                   "btnNumSeven"  : "7",
                   "btnNumEight"  : "8",
                   "btnNumNine"   : "9",
                   "btnNumZero"   : "0",
                   "btnNumPoint"  : ","};
       
   this.registerControlls();
       
};

/**
 *  set callback to be called on every value change
 *
 * @param {type} callback
 *                  the callBack function
 * @returns {undefined} */
NumericInputController.prototype.setOnValueChangedCallBack = function(callback)
{
    this._onValueChangedCallBack = callback;
}

/**
 * register popup buttons 
 * 
 * currently only to be used by NumericInputController()
 * 
 * @returns {undefined}
 */
NumericInputController.prototype.registerControlls = function()
{
    var _this = this;
    
    _this.registerControll("btnNumOne","btnNumOne");
    _this.registerControll("btnNumTwo","btnNumTwo");
    _this.registerControll("btnNumThree","btnNumThree");
    _this.registerControll("btnNumFour","btnNumFour");
    _this.registerControll("btnNumFive","btnNumFive");
    _this.registerControll("btnNumSix","btnNumSix");
    _this.registerControll("btnNumSeven","btnNumSeven");
    _this.registerControll("btnNumEight","btnNumEight");
    _this.registerControll("btnNumNine","btnNumNine");
    _this.registerControll("btnNumZero","btnNumZero");
         
    var btn = document.getElementById("btnNumPoint");
    if(btn && this._signEnabled) btn.onclick = function(){_this.decimalPoint();};
    
    var btn = document.getElementById("btnNumSign");
    if(btn && this._floatEnabled) btn.onclick = function(){_this.invertSign();};
    
    var btn = document.getElementById("btnNumAC");
    if(btn) btn.onclick = function(){_this.clear();};
    
    var btn = document.getElementById("btnNumC");
     if(btn) btn.onclick = function(){_this.back();};
            
    this._display =  document.getElementById("valueDisplayText");
    this.updateDisplay();
};

/**
 * 
 * Helper function to register a html input button to NumericInputController
 * (only for buttons that add digits)
 * 
 * @param {type} htmlID
 *                  id of html element
 * @param {type} btnID
 *                  id of NumericInputController button, see this._keyMap
 * @returns {undefined}
 */
NumericInputController.prototype.registerControll = function(htmlID, btnID)
{
   var _this = this;
   var btn = document.getElementById(htmlID);
   if(btn) btn.onclick = function(){_this.addDigit(btnID);}; 
}

/**
 * get current value
 * 
 * @returns {undefined}
 */
NumericInputController.prototype.getValue = function()
{
    return this._value;
};

/**
 * Update current value display
 * 
 * output is depening on "isMasked" param of constructor
 * 
 * @returns {undefined}
 */
NumericInputController.prototype.updateDisplay= function()
{
  
    $('#valueDisplay').removeClass('wrongpinvalue');
    if(this._isMasked)
    {
        this._display.innerText = new Array(this._value.length+1).join('*');
    }
    else if (this._symbol != undefined)
    {
        if (this._value == "") {
            this._display.innerText = "0" + this._symbol;
        } else {
            this._display.innerText = this._value + this._symbol;
        }
    }
    else {
        this._display.innerText = this._value;
    }
    
    //console.log("NumericInputController value: " + this._value);
    
    this._onValueChangedCallBack();
};

/**
 * Add a digit (or any other string) to value
 * 
 * @param {type} digit
 *                  char, digit or string to add
 * @returns {undefined}
 */
NumericInputController.prototype.addDigit = function(digit)
{
    //console.log(digit);
    if (this._value == "0" && this._symbol != undefined) {
        this._value = "";
    }
   
   if(this._value.length === this._maxChars) {
     this._value = "";
   }
   
   this._value += this._keyMap[digit];
   this.updateDisplay();
};

NumericInputController.prototype.decimalPoint = function()
{
   //console.log("decimalPoint");
   
   var pointChar = this._keyMap["btnNumPoint"];   
   
   //char limit or NOT allread existing
   if(this._value.length < this._maxChars && this._value.length > 0 && this._value.lastIndexOf(pointChar) === -1)
   {
       this._value += pointChar;
   }
   
   this.updateDisplay();   
}

/**
 * invert sign of current value
 * "-" is only added id max char limmt is not reached
 * 
 * @returns {undefined}
 */
NumericInputController.prototype.invertSign = function()
{
   //console.log("invertSign");
   
   if(this._value[0] !== "-")
   {
      if(this._value.length < this._maxChars)
      {
         this._value = "-" + this._value ;
      }
   }
   else
   {
       this._value = this._value.slice(1, this._value.length);
   }

   this.updateDisplay();
};

/**
 * delete last char of current value
 * 
 * @returns {undefined}
 */
NumericInputController.prototype.back = function()
{
    //console.log("back");
    
    this._value = this._value.slice(0, this._value.length -1);
    
    this.updateDisplay();
};


/**
 * clear current value to empty string
 * 
 * @returns {undefined}
 */
NumericInputController.prototype.clear = function()
{
    //console.log("clear");
    this._value = "";
    
    this.updateDisplay();
    
};

