(function () {

    /**
 * @enum supported connection events
 */
    ComClient.prototype.connectionEvents = {
        CONNECTING: 0,
        CONNECTED: 1,
        DISCONNECTING: 2,
        DISCONNECTED: 3,
        CONNECTION_FAILED: 4,
        REBUILD: 5
    };

    /**
     * @enum supported connection states
     */
    ComClient.prototype.connectionStates = {
        CONNECTING: 6,
        CONNECTED: 7,
        DISCONNECTING: 8,
        DISCONNECTED: 9
    };

    /**
     * Provides higher level api for connections & commands
     * 
     * @constructor
     * @implements {connectorCallback}
     * 
     * @param {string} host
     *          host you want to connect to
     * @param {string} port
     *          port of the host you use
     */
    function ComClient(host, port) {
        var self = this;
        if (host === "undefined" || port === "undefined")
            this._connector = new tsx.GdsConnector('ws://localhost:9192/', true);
        else
            this._connector = new tsx.GdsConnector('ws://' + host + ':' + port + '/', true);
        this._connector.setCallback(this);
        this._reset();

        this._eventHandlers = {};

        for (connectionEvent in this.connectionEvents)
            this._eventHandlers[this.connectionEvents[connectionEvent]] = $.Callbacks();

        this._connectionState = this.connectionStates.DISCONNECTED;
        this.registerEventHandler(this.connectionEvents.CONNECTED, function() {
            self.setConnectionState(self.connectionStates.CONNECTED);
        });
        this.registerEventHandler(this.connectionEvents.CONNECTING, function () {
            self.setConnectionState(self.connectionStates.CONNECTING);
        });
        this.registerEventHandler(this.connectionEvents.DISCONNECTED, function () {
            self.setConnectionState(self.connectionStates.DISCONNECTED);
        });
        this.registerEventHandler(this.connectionEvents.DISCONNECTING, function () {
            self.setConnectionState(self.connectionStates.DISCONNECTING);
        });
    };


    /**
     * Resets internal data structures
     * 
     * @private
     */
    ComClient.prototype._reset = function() {
        this._tagsAsNumber = [];
        this._tagsAsString = [];
        this._caseSets = [];
    };

    /**
     * Connect with target
     * 
     * @param {string} host
     *          host name of target
     * @param {number} port
     *          port of target
     * @param {string} username
     *          username for login
     * @param {string} password
     *          password for login
     */
    ComClient.prototype.connect = function(host, port, username, password) {
        this._tagsAsNumber = [];
        this._tagsAsString = [];
        this._connector.connect(host, port, username, password);
    };

    /**
     * Disconnects from target
     */
    ComClient.prototype.disconnect = function() {
        this._connector.disconnect();
    };


    /**
     * Unregisters all case sets comprised in caseSets
     * 
     * @param {array} caseSets
     *          all CaseSets
     */
    ComClient.prototype.unregisterAll = function(caseSets) {
        for (var i = caseSets.length - 1; i >= 0; i--) {
            var slotArray = this._caseSets[caseSets[i].getSlot()];

            if (slotArray == undefined || slotArray.length == 0)
                return;

            var index = slotArray.lastIndexOf(caseSets[i]);
            if (index >= 0) {
                slotArray.splice(index, 1);
            }
        }
    };

    /**
     * Registers all CaseSets for callback
     * 
     * @param {array} caseSets
     *          all CaseSets
     */
    ComClient.prototype.registerAll = function(caseSets) {
        for (var i = 0; i < caseSets.length; i++) {
            this.register(caseSets[i]);
        }
    };

    /**
     * Registers CaseSet for callback
     * 
     * @param {type} caseSet
     *          CaseSet you want to register
     */
    ComClient.prototype.register = function(caseSet) {
        if (typeof this._caseSets[caseSet.getSlot()] === 'undefined') {
            this._caseSets[caseSet.getSlot()] = [];
        }

        this._caseSets[caseSet.getSlot()].push(caseSet);

        if (!(typeof this._tagsAsString[caseSet.getSlot()] === 'undefined')) {
            this._evalCaseSet(caseSet, this._tagsAsString[caseSet.getSlot()]);
        }

        if (!(typeof this._tagsAsNumber[caseSet.getSlot()] === 'undefined')) {
            this._evalCaseSet(caseSet, this._tagsAsNumber[caseSet.getSlot()]);
        }
    };

    /**
     * Call if new tag value shall be processed. 
     * <ol>
     *  <li>stores value</li>
     *  <li>notifies registered callbacks</li>
     * </ol>
     * 
     * @param {number} tag
     *          number of tag
     * @param {string} value
     *          value of tag
     * @param {boolean} isString
     *          true if value is string, false otherwise
     */
    ComClient.prototype.onTagValue = function(tag, value, isString) {

        var typedValue = value;
        if (isString) {
            this._tagsAsString[tag] = typedValue;
        } else {
            typedValue = parseFloat(typedValue);
            if (!isNaN(typedValue)) {
                this._tagsAsNumber[tag] = typedValue;
            }
        }

        if (!(typeof this._caseSets[tag] === 'undefined')) {
            for (var i = 0; i < this._caseSets[tag].length; i++) {
                this._evalCaseSet(this._caseSets[tag][i], typedValue);
            }
        }
    };

    /**
     * Evaluate CaseSet value
     * 
     * @param {CaseSet} caseSet
     *          case set to check
     * @param {string/number} value
     *          value of CaseSet
     */
    ComClient.prototype._evalCaseSet = function(caseSet, value) {
        var newValue = null;
        if (caseSet.getCaseSetType() === CaseSetType.CONTENT) {
            var matchedCase = caseSet.getMatchingCaseForInput(parseFloat(value));
            if (matchedCase) {
                newValue = matchedCase.getNewValue();
            } else {
                newValue = caseSet._default;
            }
        } else if (caseSet.getCaseSetType() === CaseSetType.VALUE) {
            newValue = value;
        } else {
            console.log("UNHANDLED CaseSetType");
        }

        caseSet.onValue(newValue, caseSet._device);
    };

    /**
     * Toogles tagValue between 0 and value
     * 
     * @param {number/string} tag
     *          tag number
     * @param {number} value
     *          value to toggle to
     */
    ComClient.prototype.toggleTagValue = function(tag, value) {
        var currentValue = null;
        if (!(typeof this._tagsAsNumber[tag] === 'undefined')) {
            currentValue = parseFloat(this._tagsAsNumber[tag]);
        } else if (!(this._tagsAsString[tag] === 'undefined')) {
            currentValue = parseFloat(this._tagsAsString[tag]);
        }

        var newValue = parseFloat(value);
        if (currentValue !== 0.0) {
            newValue = 0.0;
        }

        this._connector.sendCommand(new SetValueCommand(tag, newValue));
    };

    /**
     * Send tagValue
     * 
     * @param {number} tag
     *          tag number
     * @param {number} value
     *          value of tag which should be send
     */
    ComClient.prototype.sendTagValue = function(tag, value) {
        this._connector.sendCommand(new SetValueCommand(tag, value));
    };

    /**
     * Gets the last value of a tag
     * 
     * @param {number} tag
     *          tag number
     * @return The last value received on the given tag, undefined if no value is stored
     */
    ComClient.prototype.getTagValue = function(tag) {
        var value = undefined;
        if (this._tagsAsString[tag] !== undefined)
            value = this._tagsAsString[tag];
        if (this._tagsAsNumber[tag] !== undefined)
            value = this._tagsAsNumber[tag];
        return value;
    };

    /**
     * Register a callback in the callback-list for the specified event
     * 
     * @param {event} connectionEvent 
     *          Connection event as integer (0-5)
     * @param {handler} handler 
     *          Callback function
     */
    ComClient.prototype.registerEventHandler = function(connectionEvent, handler) {
        if (connectionEvent in this._eventHandlers)
            this._eventHandlers[connectionEvent].add(handler);
        else
            console.error('Connection event: ' + connectionEvent + ' does not exist.');
    };

    /**
     * Unregister a callback from the callback-list for the specified event
     * 
     * @param {event} connectionEvent 
     *          Connection event as integer (0-5)
     * @param {handler} handler 
     *          Callback function
     */
    ComClient.prototype.unregisterEventHandler = function(connectionEvent, handler) {
        if (connectionEvent in this._eventHandlers)
            this._eventHandlers[connectionEvent].remove(handler);
        else
            console.error('Connection event: ' + connectionEvent + ' does not exist.');
    };

    /**
     * Calls all callbacks in the callback-list for the given event
     * 
     * @param {type} connectionEvent 
     *          Connection event as integer (0-5)
     * @param {type} message 
     *          Message which is passed to the callbacks
     */
    ComClient.prototype.fireEvent = function(connectionEvent, message) {
        if (connectionEvent in this._eventHandlers)
            this._eventHandlers[connectionEvent].fire(message);
        else
            console.error('Connection event: ' + connectionEvent + ' does not exist.');
    };

    /**
     * Sets the connection state
     * 
     * @param {type} connectionState 
     *          New connection state
     */
    ComClient.prototype.setConnectionState = function(connectionState) {
        this._connectionState = connectionState;
    };

    /**
     * Gets the current connection state
     * 
     * @returns {unresolved} connectionState
     *              Current connection state
     */
    ComClient.prototype.getConnectionState = function() {
        return this._connectionState;
    };
	
	/**
	* Register a callback in the callback-list for the connected event and execute if already connected 
	* 
	* @param {handler} handler 
	*          Callback function
	*/
    ComClient.prototype.registerOnConnectedEX = function(handler) {		
        if (this.connectionEvents.CONNECTED in this._eventHandlers)
		{
            this._eventHandlers[this.connectionEvents.CONNECTED].add(handler);			
			if(this.connectionStates.CONNECTED === this.getConnectionState())
			{				
				handler.call();
			}
		}	
        else
            console.error('Connection event: ' + this.connectionEvents.CONNECTED + ' does not exist.');
    };

    /**
     * Sends new value to target
     * 
     * @param {number} tag
     *          number of tag
     * @param {number/string} value
     *          value of tag
     */
    function SetValueCommand(tag, value) {
        this.tag = tag;
        this.value = value;
    }


    window.ComClient = new ComClient(getHTTPGetParameters("wsHost"), getHTTPGetParameters("wsPort"));
})();