/**
 * A CaseSet is a container for multiple {@link Case}s
 * 
 * @param slot
 *            that shall be used for evaluation
 * @param caseSetType
 *            type of operation associated with this CaseSet
 * @param communicationType
 *            to define method for slot value retrieval
 * @param device
 *            referenced device for callback usage 
 */
function CaseSet(slot, caseSetType, communicationType, device)
{
    this._cases = new Array();
    this._default;
    this._slot = slot;
    this._caseSetType = caseSetType;
    this._communicationType = communicationType;
    this._device = device;
}
;

/**
 * Adds a new {@link Case)
 * 
 * @param newCase
 */
CaseSet.prototype.addCase = function(newCase) {
    if (newCase === null) {
        throw "newCase == null";
    }
    if ($.inArray(newCase, this._cases) !== -1) {
        throw "Condition already in rule.";
    }
    this._cases.push(newCase);

    return this;
}
;

/**
 * Returns matching case for input
 * 
 * @param input
 * @return matching case or null if nothing matched
 */
CaseSet.prototype.getMatchingCaseForInput = function(input) {
    var matchedCase = null;
    $.each(this._cases, function(key, caseCandidate) {
        if (caseCandidate.matchesInput(input)) {
            matchedCase = caseCandidate;
            return false;
        }
    });
    return matchedCase;
}
;

/**
 * Returns the slot associated with this instance
 * 
 * @return slot
 */
CaseSet.prototype.getSlot = function() {
    return this._slot;
}
;

/**
 * Returns the {@link CaseSetType} associated with this instance
 * 
 * @return {@link CaseSetType}
 */
CaseSet.prototype.getCaseSetType = function() {
    return this._caseSetType;
}
;

/**
 * Returns the {@link CommunicationType} associated with this instance
 * 
 * @return {@link CommunicationType}
 */
CaseSet.prototype.getCommunicationType = function() {
    return this._communicationType;
}
;

/**
 * Will be called on new values
 * 
 * @param value
 *          the new value
 * @param device
 *          the referenced device
 */
CaseSet.prototype.onValue = function(value, device) {
}
;

/**
 * Sets function on value
 * 
 * @param onValueFunction
 * @returns {CaseSet.prototype}
 */
CaseSet.prototype.setOnValueFunction = function(onValueFunction) {
    this.onValue = onValueFunction;
    return this;
}
;

/**
 * set default value for no matching case
 * 
 * @param {type} defaultValue
 * @returns {CaseSet.prototype}
 */
CaseSet.prototype.setDefaultValue = function(defaultValue) {
    this._default = defaultValue;
    return this;
}
;


/** 
 * @enum Case set type
 */
CaseSetType = {
    COLOR: 0,
    CONTENT: 1,
    VALUE: 2,
    VISIBLE: 3,
    ENABLED: 4,
    WIDTH: 5,
    HEIGHT: 6,
    LEFT: 7,
    RIGHT: 8,
    TOP: 9,
    BOTTOM: 10,
    CLIP_TOP: 11,
    CLIP_BOTTOM: 12,
    CLIP_LEFT: 13,
    CLIP_RIGHT: 14,
    SELECTION: 15,
    TIMER: 16
}
;

/** 
 * @enum Case communication type 
 */
CaseCommunicationType = {
    NONE: 0,
    TAG: 1,
    TIMER: 2,
    PLUGIN: 3
}
;