/*jslint browser: true*/
/*global ko, $, _, NavbarController*/

(function (ns) {
    "use strict";
    var Log = window.Gira.Log;

    function WeatherStationSetting(saveSettingsCallbackFunc, deleteSettingsCallbackFunc, sortSettingsCallbackFunc, eulaAcceptedCallbackFunc, theConfig) {
        var saveSettingsCallback = saveSettingsCallbackFunc,
            deleteSettingsCallback = deleteSettingsCallbackFunc,
            sortSettingsCallback = sortSettingsCallbackFunc,
            eulaAcceptedCallback = eulaAcceptedCallbackFunc,
            dataService,
            configuredItems = ko.observableArray(),
            settingsItems = [],
            toggledItems = ko.observable(false),
            inEditMode = ko.observable(false),
            countOfSelectedItems = ko.observable(0),
            isDeleteEnabled = ko.observable(false),
            isNewItemAllowed = ko.observable(true),
            isEditModelAllowed = ko.observable(true),
            searchTimeout = 20000,
            isEulaAccepted = ko.observable(false),
            eulaText = ko.observable(""),
            isApplyEulaButtonEnabled = ko.observable(false),
            errorText = ko.observable(''),
            isEulaChecked = ko.observable(false),
            config = theConfig,
            searchTimeoutHandle;

        /*jslint unparam:true*/
        function errorHandler(ex, data, errorObservable) {
            try {
                if (data.response.error.text.toLowerCase().indexOf("timeout") > -1) {
                    errorObservable(config.text.weatherTimeoutMsg);
                } else {
                    errorObservable(config.text.weatherServiceUnavailableMessageText);
                }
            } catch (e) {
                Log.exception(e);
            }
        }
        /*jslint unparam:false*/

        function CountryItem(saveDataCallback) {
            var saveDataHandler = saveDataCallback,
                countries = ko.observableArray(),
                selectedCountry = ko.observable(),
                searchString = ko.observable(),
                toggled = ko.observable(false),
                isInSearch = ko.observable(false);

            function WeatherStationItem(countryId, country, saveCallback) {
                var saveData = saveCallback,
                    selectedWeatherStationGuid = ko.observable(),
                    weatherStations = ko.observableArray(),
                    allWeatherStations = [],
                    pagesCount = ko.observable(0),
                    pageIndex = ko.observable(-1),
                    maxItemsPerPage = 10,
                    weatherStationSearchString = ko.observable(),
                    toggledStation = ko.observable(false),
                    isInSearchStation = ko.observable(false),
                    searchStationTimeoutHandle,
                    showNextStations,
                    weatherStationErrorText = ko.observable(''),
                    pager = {next: function () { showNextStations(false); },
                             prev: function () { showNextStations(true); },
                             isPrevEnabled: ko.computed(function () { return pageIndex() > 0; }),
                             isNextEnabled: ko.computed(function () { return (pageIndex() + 1) * maxItemsPerPage < allWeatherStations.length; }),
                             isPager: true,
                             pagesCount: pagesCount,
                             currentPageIndex: ko.computed(function () { return pageIndex() + 1; })
                        };

                function weatherStationErrorHandler(ex, data) {
                    clearTimeout(searchStationTimeoutHandle);
                    isInSearchStation(false);
                    errorHandler(ex, data, weatherStationErrorText);
                }

                showNextStations = function showNextStations(backwards, first) {
                    weatherStations.removeAll();
                    selectedWeatherStationGuid(undefined);

                    if (backwards) {
                        if (pager.isPrevEnabled()) {
                            pageIndex(pageIndex() - 1);
                        }
                    } else {
                        if (!first) {
                            if (pager.isNextEnabled()) {
                                pageIndex(pageIndex() + 1);
                            }
                        } else {
                            pageIndex(0);
                        }
                    }
                    _.first(_.rest(allWeatherStations, pageIndex() * maxItemsPerPage), maxItemsPerPage).forEach(function (item) {
                        weatherStations.push(item);
                    });
                    if (allWeatherStations.length > maxItemsPerPage) {
                        weatherStations.push(pager);
                    }

                    toggledStation(!toggledStation());
                };

                function filterDataByFont(data) {
                    var filteredData = _.select(data, function (item) {
                        if (!item.place) {
                            return true;
                        }
                        // Filter out if start with unicode character greater then x4E00
                        return item.place.charCodeAt(0) < 19968;
                    });

                    return filteredData;
                }

                function transformData(data) {
                    var groupName,
                        group,
                        groups = _.groupBy(data, function (item) { return item.place; });

                    function makeNewName(item) {
                        if (typeof item.region === "string") {
                            item.place = item.place + " (" + item.region + ")";
                        }
                    }

                    for (groupName in groups) {
                        if (groups.hasOwnProperty(groupName)) {
                            group = groups[groupName];
                            if (group.length > 1) {
                                group.forEach(makeNewName);
                            }
                        }
                    }
                    return data;
                }

                function recreateWeatherStations(dataObj) {
                    allWeatherStations = transformData(filterDataByFont(dataObj));

                    pagesCount(Math.ceil(allWeatherStations.length / maxItemsPerPage));

                    weatherStations.removeAll();
                    selectedWeatherStationGuid(undefined);

                    if (dataObj.length === 0) {
                        weatherStationErrorText(config.text.weatherNoCountriesFoundMsg);
                    }

                    showNextStations(false, true);
                }

                function setIsInSearch() {
                    isInSearchStation(true);
                    toggledStation(!toggledStation());
                }

                function search() {
                    var query = weatherStationSearchString();
                    if (!dataService) {
                        weatherStationErrorText(config.text.weatherServiceUnavailableMessageText);
                        return;
                    }

                    if (isInSearchStation()) {
                        return;
                    }

                    if (query) {
                        setIsInSearch();
                        pageIndex(-1);
                        allWeatherStations = [];
                        weatherStations.removeAll();
                        searchStationTimeoutHandle = setTimeout(function () {
                            isInSearchStation(false);
                            weatherStationErrorText(config.text.weatherTimeoutMsg);
                        }, searchTimeout);
                        weatherStationErrorText('');
                        dataService.getWeatherStations(countryId, query, function (dataObj) {
                            clearTimeout(searchStationTimeoutHandle);
                            if (dataObj) {
                                isInSearchStation(false);
                                recreateWeatherStations(dataObj);
                            }
                        }, weatherStationErrorHandler);
                    }
                }

                function getSelectedWeatherStationGuid() {
                    return _.find(weatherStations(), function (item) {
                        return (item.guid && item.guid === selectedWeatherStationGuid());
                    });
                }

                function saveWeatherStation() {
                    var weatherStation = getSelectedWeatherStationGuid();
                    saveData(countryId, weatherStation.placeId, weatherStation.place,  weatherStation.guid);
                    countries.removeAll();
                    searchString("");
                }

                function isValid() {
                    return getSelectedWeatherStationGuid() !== undefined;
                }

                function isValidForSearch() {
                    var query = weatherStationSearchString();
                    return weatherStationSearchString() !== undefined && query.length >= 3;
                }

                return {
                    countryId: countryId,
                    country: country,
                    weatherStations: weatherStations,
                    selectedWeatherStationGuid: selectedWeatherStationGuid,
                    searchString: weatherStationSearchString,
                    ok: saveWeatherStation,
                    cancel: function () {
                        weatherStationErrorText('');
                        return undefined;
                    },
                    search: search,
                    isValid: isValid,
                    isValidForSearch: isValidForSearch,
                    isInSearch: isInSearchStation,
                    toggled: toggledStation,
                    errorText: weatherStationErrorText,
                    chooseTemplate: function (item) {
                        return item.isPager ? 'weather-settings-pager-template' : 'weather-settings-weatherstation';
                    }
                };
            }

            function saveData(countryId, weatherStationId, label, guid) {
                saveDataHandler({ countryId: countryId, weatherStationId: weatherStationId, label: label, guid: guid});
            }

            function createWeatherStationItem(countryId, country) {
                return new WeatherStationItem(countryId, country, saveData);
            }
            function recreateCountries(dataObj) {
                countries.removeAll();
                selectedCountry(undefined);

                if (dataObj.length === 0) {
                    errorText(config.text.weatherNoCountriesFoundMsg);
                }

                dataObj.forEach(function (item) {
                    countries.push(createWeatherStationItem(item.countryId, item.country));
                });
                toggled(!toggled());
            }

            function setIsInSearch() {
                isInSearch(true);
                toggled(!toggled());
            }

            function countryErrorHandler(ex, data) {
                clearTimeout(searchTimeoutHandle);
                isInSearch(false);
                errorHandler(ex, data, errorText);
            }

            function search() {
                var query = searchString();
                if (!dataService) {
                    errorText(config.text.weatherServiceUnavailableMessageText);
                    return;
                }

                if (isInSearch()) {
                    return;
                }
                if (query) {
                    errorText('');
                    setIsInSearch();
                    countries.removeAll();
                    searchTimeoutHandle = setTimeout(function () {
                        isInSearch(false);
                        errorText(config.text.weatherTimeoutMsg);
                    }, searchTimeout);

                    dataService.getCountries(query, function (dataObj) {
                        clearTimeout(searchTimeoutHandle);
                        if (dataObj) {
                            isInSearch(false);
                            recreateCountries(dataObj);
                        }
                    }, countryErrorHandler);
                }
            }

            function getSelectedCountry() {
                return _.find(countries(), function (item) {
                    return (item.countryId === selectedCountry());
                });
            }

            function isValid() {
                return getSelectedCountry() !== undefined;
            }

            function isValidForSearch() {
                var query = searchString();
                return searchString() !== undefined && query.length >= 2;
            }

            return {
                createWeatherStationItem: createWeatherStationItem,
                searchString: searchString,
                countries: countries,
                selectedCountry: selectedCountry,
                getSelectedCountry: getSelectedCountry,
                search: search,
                cancel: function () {
                    errorText('');
                    countries.removeAll();
                    searchString("");
                },
                isValid: isValid,
                isValidForSearch: isValidForSearch,
                isInSearch: isInSearch,
                toggled: toggled
            };
        }

        function checkIfEditModeAllowed() {
            if (configuredItems().length > 0) {
                isEditModelAllowed(true);
            } else {
                isEditModelAllowed(false);
                inEditMode(false);
            }
        }

        function saveData(data) {
            saveSettingsCallback(data);
        }

        function removeSelections() {
            configuredItems().forEach(function (item) {
                item.isSelected(false);
            });
        }

        function checkIsDeleteAllowed(value) {
            if (value) {
                countOfSelectedItems(countOfSelectedItems() + 1);
            } else {
                countOfSelectedItems(countOfSelectedItems() - 1);
            }

            if (countOfSelectedItems() > 0) {
                isDeleteEnabled(true);
            } else {
                isDeleteEnabled(false);
            }
        }

        function removeAllItems() {
            settingsItems = [];
            configuredItems.removeAll();
            countOfSelectedItems(0);
            isDeleteEnabled(false);
        }

        function createListItem(weatherStationId, label, guid) {
            var obj = {weatherStationId: weatherStationId, label: label, guid: guid, isSelected: ko.observable(false)};
            obj.isSelected.subscribe(checkIsDeleteAllowed);
            return obj;
        }

        function checkEula(data) {
            isEulaAccepted(data.acceptedLicense);
            if (!isEulaAccepted() && eulaText.length === 0) {
                eulaText(config.eulaI18n.eula);
            } else if (isEulaAccepted() && eulaText.length > 0) {
                eulaText("");
            }
        }

        function applyData(dataObj) {
            checkEula(dataObj);
            removeAllItems();

            dataObj.weatherStations.forEach(function (item) {
                settingsItems.push(item);
                configuredItems.push(createListItem(item.weatherStationId, item.label, item.guid));
            });
            toggledItems(!toggledItems());
            if (configuredItems().length >= 5) {
                isNewItemAllowed(false);
            } else {
                isNewItemAllowed(true);
            }

            checkIfEditModeAllowed();
        }

        function okResult(goBack) {
            if (inEditMode()) {
                inEditMode(false);
            } else {
                goBack();
            }
            removeSelections();
        }

        function deleteItems() {
            var itemsForDelete = [];
            configuredItems().forEach(function (item) {
                if (item.isSelected()) {
                    itemsForDelete.push(item);
                }
            });
            deleteSettingsCallback(itemsForDelete);
        }

        function goInEditMode() {
            inEditMode(true);
        }

        function saveNewSortOrder(orderedItems) {
            var index,
                settingItem,
                orderedForSave = [];

            function getSettingsItem(orderedItem) {
                return _.find(settingsItems, function (item) {
                    return !!(orderedItem.weatherStationId === item.weatherStationId && orderedItem.guid === item.guid);
                });
            }

            for (index = 0; index < orderedItems.length; index = index + 1) {
                settingItem = getSettingsItem(orderedItems[index]);
                orderedForSave.push(settingItem);
            }

            sortSettingsCallback(orderedForSave);
        }

        function applyEula() {
            var license,
                swiperContainer,
                swiper;

            try {
                eulaAcceptedCallback();
                isEulaAccepted(true);

                license = $("#weather-license-top");
                swiperContainer = license.closest(".settings-swiper-container");
                swiper = swiperContainer.data("swiper");
                if (swiper && swiper.params && !swiper.params.onSetWrapperTransform) {
                    swiper.removeCallbacks("SetWrapperTransform");
                }
            } catch (ex) {
                Log.exception(ex);
            }
        }

        function getEulaText() {
            if (eulaText() === "") {
                checkEula({acceptedLicense: false});
            }
            return eulaText;
        }

        function setDataService(dataServiceRest) {
            dataService = dataServiceRest;
        }

        isEulaChecked.subscribe(function (value) {
            isApplyEulaButtonEnabled(value);
        });

        return {
            apply: function () { return undefined; },
            applyData: applyData,
            configuredItems: configuredItems,
            countrySearch: new CountryItem(saveData),
            isNewItemAllowed: isNewItemAllowed,
            isEditModelAllowed: isEditModelAllowed,
            inEditMode: inEditMode,
            deleteItems: deleteItems,
            goInEditMode: goInEditMode,
            ok: function () {
                return okResult;
            },
            cancel: function () {
                isEulaChecked(false);
                return undefined;
            },
            applyEula: applyEula,
            isDeleteEnabled: isDeleteEnabled,
            toggled: toggledItems,
            saveNewSortOrder: saveNewSortOrder,
            isEulaAccepted: isEulaAccepted,
            getEulaText: getEulaText,
            isApplyEulaButtonEnabled: isApplyEulaButtonEnabled,
            errorText: errorText,
            isEulaChecked: isEulaChecked,
            setDataService: setDataService
        };
    }

    ns.WeatherStationSetting = WeatherStationSetting;
}(window.Gira.Weather));

//# sourceURL=weather.settings.weatherstation.js