﻿/**
* JavaScript based GDS API Library. Copyright 2012 (c) Gira, Giersiepen GmbH & Co. KG. All rights reserved.
* @version 1.0
* @author ise GmbH
*/

/**
* Helper functions
* (5.10.13)
*/

/**
* The gdsQuery instance of gdsLibrary
*/
(function () {
    var gdsQuery = function () {
        return new gdsLibrary();
    };

    var gdsLibrary = function () {
    };

    var ws = null;
    var wsUri;
    var eventHandlers = $.Callbacks();
    var eventTypeHandlers = {
        Value: $.Callbacks(),
        Configuration: $.Callbacks(),
        Discovery: $.Callbacks()
    }
    var readyHandlers = $.Callbacks();
    var onResponseHandlerConfig = {};
    var urnMapping = {};
    var deviceConfiguration = null;
    var deviceConfigurationIpc = null;
    var gdsqueryId = 1;
    var maxRequests = 1000;
    var socketEventHandlers = {
        open:$.Callbacks(),
        error:$.Callbacks(),
        close:$.Callbacks(),
        message: $.Callbacks()
    }

    gdsLibrary.prototype = {
        /* Initializes the urn mapping and datapoint values*/
        init: function () {
            onResponseHandlerConfig = {};
            urnMapping = {}
            this.request({ "request": { "command": "GetProcessView" } }, function (response) {

                function removeDeviceName(urn) {
                    return urn.replace(/[^:.]*\./, "");
                }

                var processView = response.processView;
                if (processView) {
                    processView.datapoints.forEach(function (datapoint) {
                        eventTypeHandlers.Value.fire({ value: { id: datapoint.id, urn: datapoint.urn, 'new': datapoint.state } });
                        urnMapping[removeDeviceName(datapoint.urn)] = datapoint.id;
                    });
                    processView.channels.forEach(function (channel) {
                        urnMapping[removeDeviceName(channel.urn)] = channel.id;
                    });
                }
                readyHandlers.fire();
            });
        },
        /* register ready handler */
        ready: function (handler) {
            readyHandlers.add(handler);
        },
        /* get device configuration */
        deviceConfig: function () {
            return deviceConfiguration;
        },
        /* get device ipc configuration */
        deviceConfigIpc: function () {
            return deviceConfigurationIpc;
        },
        /* register callbacks for soecket events (open, close, error, message) */
        registerSocketEventHandlers: function(handlers) {
            for(var eventType in socketEventHandlers) {
                socketEventHandlers[eventType].add(handlers[eventType]);
            }
        },
        /* opens a websocket connection to the specified URI if no connection is active yet */
        connect: function (uri) {
            var self = this;
            wsUri = uri;
            if (!ws) {
                ws = new WebSocket(uri);
                ws.onopen = function (ea) { self.onOpen(ea); };
                ws.onclose = function (ea) { self.onClose(ea);};
                ws.onerror = function (ea) { self.onError(ea); };
                ws.onmessage = function (ea) { self.onMessage(ea); };
            } 
        },
        /* triggers a reconnect to the current websocket target */
        reconnect: function () {
            this.close();
            this.connect(wsUri);
        },
        /* retruns true if the websocket is connected, false otherwise */
        isConnected: function() {
            return (ws != null && ws.readyState == 1);
        },
        /* sends message */
        send: function (message) {
            try {
                ws.send(message);
            } catch (e) {}
        },
        /* close the current websocket connection */
        close: function () {
            if (ws) {
                ws.close();
                delete ws;
                ws = null;
            }
        },
        /* send registration command to the gdsApi */
        register: function () {
            var req = {
                request: {
                    command: 'registerUI'
                }
            };
            this.send(JSON.stringify(req));
        },
        /* sends a request to the gdsApi and stores the according response handler */
        request: function (req, responseHandler) {
            if (req && req.request) {
                if (responseHandler) {
                    onResponseHandlerConfig[gdsqueryId] = responseHandler;
                    req.request._gdsqueryId = gdsqueryId;
                    gdsqueryId = (gdsqueryId + 1) % maxRequests;
                }
                this.send(JSON.stringify(req));
            }
        },
        /* sets a gds app value */
        setAppValue: function (appName, key, val) {
            var req = {
                request: {
                    command: 'SetAppValue',
					appName: appName,
                    key: key,
                    value: val
                }
            };
            this.send(JSON.stringify(req));
        },
        /* send request for a gds app value */
        getAppValue: function (appName, key, responseHandler) {
            var req = {
                request: {
                    command: 'GetAppValue',
                    appName: appName,
                    key: key
                }
            };
            gdsLibrary.prototype.request(req, responseHandler);
        },
        /* delete gds app value */
        deleteAppValue: function (appName, key) {
            var req = {
                request: {
                    command: 'DeleteAppValue',
                    appName: appName,
                    key: key
                }
            };
            this.send(JSON.stringify(req));
        },
        /* sets the value of a datapoint */
		setValue: function (id, val) {
            var req = {
                request: {
                    command: 'SetValue',
                    id: id,
                    value: val
                }
            };
            this.send(JSON.stringify(req));
		},
        /* sets the value of a datapoint and registers response handler*/
		setValueWithFeedback: function (id, val, responseHandler) {
		    var req = {
		        request: {
		            command: 'SetValue',
		            id: id,
		            value: val
		        }
		    };
		    this.request(req, responseHandler);
		},
        /* gets the value of a datapoint */
        getValue: function (id) {
            var req = {
                request: {
                    command: 'GetValue',
                    id: id,
                    verbose: 'true'
                }
            };
            this.send(JSON.stringify(req));
        },
        /* updates the device configuration */
        updateDeviceConfiguration: function (isIpc, callback, extraParams) {
            var req = isIpc ? { request: { command: 'GetDeviceConfig', ipc: 'true' } }
                            : { request: { command: 'GetDeviceConfig', ipc: 'false', data: (extraParams ? extraParams : ''), pretty: 'true' } };
            if (isIpc == true) {
                this.request(req, function (response) {
                    deviceConfigurationIpc = response.deviceConfig.ipc;
                    if (callback) callback(deviceConfigurationIpc);
                });
            } else {
                this.request(req, function (response) {
                    deviceConfiguration = response.deviceConfig;
                    if (callback) callback(deviceConfiguration);
                });
            };
        },

        /* sets a device configuration entry */
        setDeviceConfigurationValue: function (isIpc,configElement,callback) {
            var ipc = isIpc ? 'true' : 'false';
            var req = {
                request: {
                    command: 'SetDeviceConfig',
                    ipc: ipc,
                    deviceConfig: configElement
                }
            };
            this.request(req, function (response) {
                try {
                    if (isIpc) {
                        var deviceConfigurationIpc = response.deviceConfig.ipc;
                        if (callback) callback(deviceConfigurationIpc);
                    } else {
                        var deviceConfiguration = response.deviceConfig;
                        if (callback) callback(deviceConfiguration);
                    }
                } catch (e) {
                    if (callback) callback();
                }
            });
        },
        /* add handler for value changes */
        addValueHandler: function (eventHandler) {
            eventTypeHandlers.Value.add(eventHandler);
        },
        /* add handler for configuration changes */
        addConfigurationHandler: function (eventHandler) {
            eventTypeHandlers.Configuration.add(eventHandler);
        },
        /* add handler for discovery events */
        addDiscoveryHandler: function (eventHandler) {
            eventTypeHandlers.Discovery.add(eventHandler);
        },
        /* websocket onopen handler */
        onOpen: function (ea) {
            this.register();
            this.updateDeviceConfiguration(false); // w/o Ipc structure
            this.updateDeviceConfiguration(true); // incl. Ipc structure
            this.init();
            socketEventHandlers.open.fire(ea);
        },
        /* websocket onclose handler */
        onClose: function (ea) {
            if (ws) {
                this.close();
                var self = this;
                setTimeout(function () {
                    self.connect(ea.currentTarget.url);
                }, 10000);
            }
            socketEventHandlers.close.fire(ea);
        },
        /* websocket onerror handler */
        onError: function (ea) {
            socketEventHandlers.error.fire(ea);
        },
        /* websocket onmessage*/
        onMessage: function (ea) {
            if (ea.data) {
                var msg = JSON.parse(ea.data);
                if (msg) {
                    /* handle response messages */
                    if (msg.response) {
                        if (msg.response.request) {
                            if (msg.response.request._gdsqueryId) {
                                var queryId = msg.response.request._gdsqueryId;
                                if (onResponseHandlerConfig[queryId]) {
                                    onResponseHandlerConfig[queryId](msg.response);
                                    delete onResponseHandlerConfig[queryId];
                                } 
                            }
                        }
                    /* handle event messages */
                    } else if (msg.event) {
                        eventHandlers.fire(msg);
                        var type = msg.event.type;
                        var callbacks = eventTypeHandlers[type];
                        if (callbacks) callbacks.fire(msg.event);

                    } else {
                        console.log('unknown message');
                    }
                } else {
                    console.log('invalid message');
                }
            }
            socketEventHandlers.message.fire(ea);
        },
        /* send get request for a configuration object */
        getConfigById: function(id, responseHandler) {
            var req = {
                request: {
                    command: 'GetConfiguration',
                    object: {
                        id: id
                    }
                }
            };
            gdsLibrary.prototype.request(req, responseHandler);         
        },
        emitConfigurationChangeBegin: function() {
            var req = {
                request: {
                    command: 'EmitEvent',
                    eventType: 'configurationChangeBegin'
                }

            };
            this.send(JSON.stringify(req));
        },
        emitConfigurationChangeEnd: function() {
            var req = {
                request: {
                    command: 'EmitEvent',
                    eventType: 'configurationChangeEnd'
                }

            };
            this.send(JSON.stringify(req));
        },
        disableConfigRange: function (startId, endId) {
            var req = {
                request: {
                    command: 'SetConfiguration',
                    range: {
                        from: startId.toString(),
                        to: endId.toString(),
                        enabled: "false"
                    }
                }

            };
            this.send(JSON.stringify(req));
        },
        
        /* sets a configuration object */
        setConfigObject: function(object) {
            var req = {
                request: {
                    command: 'SetConfiguration',
                    object: object
                }

            };
            this.send(JSON.stringify(req));
        },
        /* returns the id associated to the specified urn */
        getId: function(urn) {
            return urnMapping[urn];
        },

        /* write a log entry to the GDS log */
        writeLog: function(context, filename, level, message) {
            var req = {
                request: {
                    command: "WriteLogEntry",
                    context: context,
                    filename: filename,
                    level: level,
                    message: message,
                    pretty: "true"
                }
            };

            this.send(JSON.stringify(req));
        }

        }; // End of gdsLibrary        

    window.gds$ = gdsQuery();
})(); // End of anonymous function
