/**
 * The Class tsx.ErrorHandler handles the error management
 * 
 */
tsx.ErrorHandler = (function () {

    var handler = {}
    var mItems = new Array();
    var mSuperChannelItems = new Array();

    /**
     * register a new error item (world, network, etc)
     * which can send error Messages
     *
     * @param {string} itemId 
     *          The Error Item ID (identify for a world)
     */
    handler.register = function (itemId) {
        var existing = itemExists(itemId);
        if (!existing) {
            var item = new Object();
            item["id"] = itemId;
            item["message"] = "";
            mItems.push(item);
        }
    }

    handler.registerSuperChannel = function(message) {
        if (mSuperChannelItems.length === 0) {
            var item = new Object();
            item["id"] = 0;
            item["message"] = message;
            mSuperChannelItems.push(item);
            showErrorWarning();
            initPopup(tsx.Popup.ModeType.SUPER, 0);
            return true;
        } else {
            return false;
        }
    }
    handler.removeSuperChannel = function() {
        if (mSuperChannelItems.length > 0) {
            mSuperChannelItems = new Array();
            removeErrorFromPopup(0, tsx.Popup.ModeType.SUPER);
        }
    }
    /**
     * set an Error or Delete an Error
     *
     * @param {string} itemId 
     *          The Error Item ID (identify for a world)
     * @param {int} errorCode
     *          Error Code 1 = error
     *          Error Code 0 = no error
     * @param {string} message  
     *          Error Message to display
     */
    handler.setError = function (itemId, errorCode, message) {
        if (errorCode ===  1) {
            for (var i = 0; i < mItems.length; i++) {
                if (mItems[i]["id"] === itemId) {
                    mItems[i]["message"] = message;

                    showErrorWarning();

                    initPopup(tsx.Popup.ModeType.ERROR, i);
                    return mSuperChannelItems.length == 0;
                }
            }
        }
        else if (errorCode === 0) {
            removeError(itemId);
            return true;
        }
        return false;
    }

    /*
     * removes the Error from the List for the specified itemId
     * @param {string} itemId 
     *          The Error Item ID (identify for a world)
     */
    function removeError(isSuperChannel, itemId) {
        var isRemoveAll = (typeof itemId == "undefined");
        if (isSuperChannel == true) {
            handler.removeSuperChannel();
        } else if (isRemoveAll || itemExists(itemId)) {
            for (var i = mItems.length - 1; i >= 0; i--) {
                if (isRemoveAll || mItems[i].id === itemId) {
                    removeErrorFromPopup(mItems[i].id, tsx.Popup.ModeType.ERROR);
                    mItems.splice(i, 1);
                    if (isRemoveAll) {
                        continue;
                    } else {
                        break;
                    }
                }
            }
        }
        if (mItems.length == 0 && mSuperChannelItems.length == 0) {
            hideErrorWarning();
        }
    }

    /*
     * Request a Popup for the Errors
     */
    function initPopup(mode, index) {
        var id;
        var message;
        if (mode === tsx.Popup.ModeType.ERROR) {
            id = mItems[index]["id"];
            message = mItems[index]["message"];
        }
        else if (mode === tsx.Popup.ModeType.SUPER) {
            id = mSuperChannelItems[index]["id"];
            message = mSuperChannelItems[index]["message"];
        }

        tsx.Popup.init(mode);
        tsx.Popup.addMessage(id, mode, message);
    }

    /*
     * Shows the Error Warning Icon in the Statusbar
     */
    function showErrorWarning() {
        tsx.StatusbarController.alert(true);
    }

    /*
     * Hide the Error Warning Icon in the Statusbar
     */
    function hideErrorWarning() {
        tsx.StatusbarController.alert(false);
    }

    /*
     * Removes the Error Message from the Popup
     */
    function removeErrorFromPopup(itemId, channel) {
        tsx.Popup.removeMessage(itemId, channel);
    }

    /*
     * Checks for an existing registert ItemID
     */
    function itemExists(itemId) {
        var existing = false;

        for (var i = 0; i < Object.keys(mItems).length; i++) {
            if (mItems[i]["id"] === itemId) {
                existing = true;
            }
        }
        return existing;
    }

    //SystemMessage
    (function () {
        /* returns true if channel accepts messages; signalize state machine whether to switch state */
        function openMessage(args) {
            switch (args.messageType) {
                case "superError":
                    return handler.registerSuperChannel(args.message);
                case "error":
                    handler.register(args.itemId);
                    return handler.setError(args.itemId, 1, args.message);
                case "okay":
                    tsx.Popup.init(tsx.Popup.ModeType.OKAY);
                    tsx.Popup.addMessage(0, tsx.Popup.ModeType.OKAY, args.message);
                    break;
                case "confirmation":
                    tsx.Popup.init(tsx.Popup.ModeType.INFO, args.rightBtnCallback, args.leftBtnCallback);
                    tsx.Popup.addMessage(0, tsx.Popup.ModeType.INFO, args.message);
                    break;
                case "warning":
                    tsx.Popup.init(tsx.Popup.ModeType.WARNING);
                    tsx.Popup.addMessage(0, tsx.Popup.ModeType.WARNING, args.message);
                    break;
                default:
            }
            return true;
        }
        /* returns true if no messages left open; signalize state machine whether to switch state */
        function closeMessage(args) {
            if (typeof args == "undefined") {
                if (mSuperChannelItems.length > 0) {
                    removeError(true);
                }
                if (mItems.length > 0) {
                    removeError(false);
                }
                tsx.Popup.closePopup();
                return true;
            } else if(args){
                switch (args.messageType) {
                    case "superError":
                        removeError(true);
                        return mSuperChannelItems.length == 0 && mItems.length == 0 && tsx.Popup.isEmpty();
                    case "error":
                        removeError(false, args.itemId);
                        return mSuperChannelItems.length == 0 && mItems.length == 0 && tsx.Popup.isEmpty();
                    case "okay":
                        tsx.Popup.removeMessage(false, tsx.Popup.ModeType.OKAY);
                    case "warning":
                        tsx.Popup.removeMessage(false, tsx.Popup.ModeType.WARNING);
                    case "confirmation":
                        tsx.Popup.removeMessage(false, tsx.Popup.ModeType.INFO);
                    default:
                        tsx.Popup.closePopup(args.animationEndCallback);
                        break;
                }
            }

            return true;
        }
        function minimizeMessage(animationEndCallback) {
            tsx.Popup.closePopup(animationEndCallback);
            return true;
        }
        function maximizeMessage() {
            if (mSuperChannelItems.length == 0 && mItems.length == 0) {
                return false;
            } else {
                tsx.Popup.show();
                return true;
            }
        }
        fsm.API.SystemMessage.Init(openMessage, closeMessage, minimizeMessage, maximizeMessage);
    })();

    return handler;
}());